# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#      crsinstall.pm
#
#   DESCRIPTION
#      This module contains fresh install related functions for CRS/SIHA
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#   xyuan       12/25/12 - Fix bug 16014163 - disable listener changes for
#                          Windows
#   xyuan       12/20/12 - XbranchMerge xyuan_bug-15969912 from main
#   xyuan       12/09/12 - Fix bug 15969912
#   shiyer      11/28/12 - #15926544:XAG msg files
#   takiba      10/09/12 - bug13868167: recreate local.ocr on SIHA upgrade
#   ssprasad    11/13/12 - Disable OKA actions for 12.1
#   xyuan       10/29/12 - Fix bug 14796373 & 14827104
#   rdasari     10/29/12 - no need to pass role for first stack startup
#   rdasari     10/26/12 - createCredDomain for legacy asm
#   xyuan       10/24/12 - Fix bug 14801109
#   dpham       10/19/12 - Fix bug 14782707.
#   rdasari     10/17/12 - fix import_asm_credentials
#   rdasari     10/09/12 - create credential domains in all cases
#   rdasari     10/09/12 - update gpnp stage profile for add node
#   epineda     09/27/12 - Bugfix 13539130
#   xyuan       09/27/12 - Fix bug 14671774 - fix sub getHostVIP so that it can
#                          recognize "prefixlen6" for IPV6
#   shullur     09/25/12 - For moving getcrsrelver to crsutils.pm
#   sidshank    09/24/12 - fix bug 14620416.
#   shullur     09/17/12 - For adding check of version greater than 12 in case
#                          of bdb
#   rdasari     09/13/12 - start asm proxy only on hub nodes
#   rdasari     09/13/12 - fix copy_asm_credentials for far asm
#   xyuan       09/11/12 - Fix bug 14592705
#   xyuan       09/04/12 - Fix bug 14560280
#   xyuan       08/31/12 - Fix bug 14535011
#   ssprasad    08/23/12 - Fix bug 14528631 - add isOKASupported() check
#                          in perform_installKADriver()
#   rdasari     08/22/12 - add configure_ASM_oda
#   dpham       08/21/12 - Add XAG component
#   madoming    08/17/12 - Fix bug 14320731. Start proxy asm on all nodes using
#                          near ASM Configuration.
#   xyuan       08/08/12 - Fix bug 9584563
#   rdasari     08/01/12 - change node role to rim if hub role fails
#   rdasari     07/25/12 - do not start ohasd before CSS exclusive start
#   shmubeen    07/18/12 - add AFD installation
#   ssprasad    07/18/12 - Report msg 400 instead of 2008 on reboot.
#   xyuan       07/12/12 - Fix bug 14302401
#   xyuan       07/08/12 - Fixed an issue where the 'ROOTCRS_BOOTCFG'
#                          checkpoint is missing on non-first nodes
#   sidshank    07/03/12 - fix bug 14196853.
#   rdasari     07/03/12 - import credentials before configure OCR for Far asm
#   shullur     06/15/12 - Change the bdbloc to default in the ora file.
#   rdasari     06/13/12 - create asm credential domains for legacy asm also
#   ssprasad    06/07/12 - KA driver calls in install path
#   rdasari     06/01/12 - make root auto option default for Windows
#   xyuan       05/24/12 - Fix bug 14111446
#   sidshank    05/23/12 - fix bug 14106919
#   xyuan       05/18/12 - Fix bug 14082413 - Only start CSS in X mode on the
#                          first node
#   xyuan       05/16/12 - Fix bug 13795377
#   rdasari     05/01/12 - start ohasd before importing asm credentials
#   akhaladk    04/30/12 - undo w/q for 13983808
#   rdasari     04/26/12 - WA for bug 13983808
#   sidshank    04/20/12 - fix for bug 13926993.
#   rtamezd     04/12/12 - Fix bug 13931049
#   sidshank    04/09/12 - removing the call to redirect/restore stdout on
#                          windows.
#   rdasari     04/04/12 - bounce stack after create asm credentials
#   rtamezd     03/26/12 - Move stop_ohasd_resources to crsutils.pm
#   rdasari     03/22/12 - bounce ohasd before initial config
#   rtamezd     03/09/12 - Fix bug 13822648
#   xyuan       03/08/12 - Fix bug 13797791
#   rtamezd     03/07/12 - Fix bug 13786398
#   rtamezd     02/21/12 - Fix bug 13730374
#   shullur     02/15/12 - XbranchMerge shullur_bug-12976590 from st_has_11.2.0
#   rtamezd     02/09/12 - Fix bug 10083997
#   sidshank    02/09/12 - Remove call to first_node_tasks()
#   xyuan       02/07/12 - Moved isVersionMatch to crsutils.pm
#   rtamezd     02/07/12 - Fix bug 13643262
#   xyuan       02/07/12 - Fix bug 13691391
#   xyuan       02/05/12 - Fix bug 13405738
#   anjiwaji    01/19/12 - Fix chkpoint comparison.
#   anjiwaji    01/17/12 - Remove debug lines
#   rtamezd     01/17/12 - Moved add_localOlr_OlrConfig_OcrConfig to crsutils
#   xesquive    12/16/11 - Fix bug 13342583
#   xyuan       12/13/11 - Added setHubSize
#   xyuan       12/13/11 - Fix bug 13480148
#   xyuan       12/06/11 - Fix bug 12863550
#   xesquive    12/05/11 - Create gridhome domain credentials in OCR
#   agraves     11/30/11 - Add functionality to check for reboot necessary in
#                          ACFS driver install.
#   rdasari     11/28/11 - stop ohasd before stack startup on rim nodes
#   sidshank    11/21/11 - fix for bug 13416034
#   xyuan       11/20/11 - Fix bug 13404894
#   sidshank    11/15/11 - adding backupOCR subroutine
#   xyuan       11/14/11 - Fix bug 13340630
#   xesquive    11/07/11 - Not start/stop ASM while adding a node
#   xyuan       11/05/11 - Fix bug 13329109
#   xyuan       11/02/11 - Fix bug 13251040
#   xyuan       10/31/11 - Fix bug 13328777
#   xyuan       10/20/11 - XbranchMerge ksviswan_bug-12630162 from
#                          st_has_11.2.0.3.0
#   xyuan       10/05/11 - Fix bug 13066014
#   rdasari     09/07/11 - create asm credentials after configNode
#   xyuan       09/07/11 - Fix bug 12908387
#   xesquive    08/15/11 - forward merge from bug 12587677
#   xyuan       08/10/11 - Far ASM support
#   xyuan       08/09/11 - Fix bug 12843894
#   xyuan       08/03/11 - Fix bug 11851866
#   xyuan       07/27/11 - XbranchMerge xyuan_bug-12701521 from
#                          st_has_11.2.0.3.0
#   xesquive    07/26/11 - Add functions set_bold and reset_bold instead of
#                          print color
#   xyuan       07/19/11 - BC commands for 12c
#   jkellegh    07/15/11 - Root.sh should not call asmca for add node
#   nkorehis    07/03/11 - bug-12710656:fix get_has_version
#   lmortime    06/27/11 - Make EVMD start first
#   rdasari     06/16/11 - near ASM support
#   dpham       05/23/11 - Modulize upgrade function
#   dpham       03/28/11 - New for 12c
#

package crsinstall;

use strict;
use English;
use Exporter;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Sys::Hostname;
use POSIX qw(tmpnam);
use Carp;
use Socket;
use Config;
use Env qw(NLS_LANG);

# root script modules
use crsutils;
use crsgpnp;
use crsohasd;
use oracss;
use oraacfs;
use crska;
use oraafd;
use s_crsutils;

#
use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func  = qw(create_starting_ckpt_entry precheck save_param_file backupOLR
                   precheck_siha check_CRSConfig setup_env isOLRConfigured 
                   local_only_config_exists start_or_upgrade_siha 
                   create_IPDOS_config create_ohasd_resources perform_installUSMDriver
                   initial_cluster_validation configureCvuRpm configureRemoteNodes
                   perform_initial_config createLocalOnlyOCR backupOCR 
                   perform_installKADriver install_xag
                  );

push @EXPORT, @exp_func;

sub new {
   shift;
   crsutils->new(@_);

   # Put a null string in for VF discover string so that the change
   # will not be rejected (bug 7694835)
   $CFG->VF_DISCOVERY_STRING('');

   if ((! $CFG->ASM_STORAGE_USED) &&
       (! $CFG->SIHA) &&
       ($CFG->defined_param('VOTING_DISKS')))
   {
      $CFG->VF_DISCOVERY_STRING($CFG->params('VOTING_DISKS'));
   }

   # initialize $CFG->VF_DISCOVERY_STRING
   if ($CFG->ASM_STORAGE_USED) {
      # Put a null string in for VF discover string so that the change
      # will not be rejected (bug 7694835)
      $CFG->VF_DISCOVERY_STRING('');
   }
   else {
      if (! $CFG->SIHA) {
         if ($CFG->defined_param('VOTING_DISKS')) {
            $CFG->VF_DISCOVERY_STRING($CFG->params('VOTING_DISKS'));
         }
         else {
            $CFG->VF_DISCOVERY_STRING('');
         }
      }
   }

   if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # redirect stdout/stderr as appropriate for Windows only
      s_redirect_souterr($CFG->crscfg_trace_file . "_OUT");
   }
   rscPreChecks();

   if ($CFG->init)
   {
     InitEnv();
   }
   # fresh install
   elsif ($CFG->SIHA)
   {
      HAInstall();
   }
   else {
      CRSInstall();
   }

   if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # restore stdout/stderr as appropriate
      s_restore_souterr();
   }
}

sub InitEnv
{
   # create dirs & set permissions
   my @crsconfig_dirs = read_file(catfile($CFG->ORA_CRS_HOME, 'crs',
                                          'utl', 'crsconfig_dirs'));
   create_dirs(\@crsconfig_dirs);

   my @crsconfig_fileperms = read_file(catfile($CFG->ORA_CRS_HOME, 'crs',
                                               'utl', 'crsconfig_fileperms'));
   set_file_perms(\@crsconfig_fileperms);

   # Set owner/group of ORA_CRS_HOME and its parent dir to root/dba
   if (! is_dev_env() && (! $CFG->SIHA) &&
      ($CFG->platform_family eq "unix"))
   {
      s_setParentDirOwner ($CFG->SUPERUSER, $CFG->ORA_CRS_HOME);
   }
}

sub CRSInstall
{
   my $success = TRUE;

   # instantiate scripts
   instantiate_scripts();

   # Before File Permissions module is invoked, we need to add entries for
   # OCR and Voting Disk locations to crsconfig_fileperms (Bug 8236090).
   add_localOlr_OlrConfig_OcrConfig();

   # run directory creation, files creation/permissions modules
   setup_env();

   create_starting_ckpt_entry();

   precheck();

   # validate network interface
   if (! isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')) &&
      (! isInterfaceValid()))
   {
      writeCkpt("ROOTCRS_STACK", CKPTFAIL);
      exit 1;
   }

   save_param_file();

   osd_setup();

   # on NT, set_perms_ocr_vdisk() function cannot be executed until
   # s_osd_setup() is finished
   set_perms_ocr_vdisk();

   prepare_to_start_clusterware();

   if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST'))) {
      add_clscfg();
   }

   # configure node
   $success = perform_configNode();

   # create asm credentials after the asm resource is created
   # the asm resource gets created during configNode above
   # the OLR domain is created on all nodes
   if (FAILED == createCredDomain('ASM', 'OLR') )
   {
     die(dieformat(377));
   }
   create_asm_credentials();

   if(isFirstNodeToStart())
   {
       # creates a GridHome credentials in OCR
       createCredDomain('GRIDHOME', 'OCR')  || die(dieformat(380));

       if (isNearASM())
       {
         trace("add asm-proxy for near asm");
         srvctl(TRUE, "add asm -proxy") || die(dieformat(372));

         # needed to enable remote asm (bug 13899801)
         stopFullStack("force") || die(dieformat(349));
         startFullStack() || die(dieformat(117));
       }
   }

   # Workaround fix for bug 8309620 and 9954172 and 10056987.
   # The start of asm resource pulled up acfs resource due to
   # startup dependency , but didn't get stopped as part of asm stop.
   if (isFirstNodeToStart() &&
       !isOCRonASM() &&
       !isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
   {
      start_asm();
      stop_asm();
      stop_resource("ora.registry.acfs", "-f");
   }

   # The state of resource ora.asm is INTERMEDIATE even
   # after 'crsctl start crs -wait' returns with 0.
   #
   # On the first node, the autostart of ora.proxy_advm
   # won't be successful until ora.asm comes to ONLINE.
   if (isFirstNodeToStart() && isNearASM() && isHubNode())
   {
     for (my $i = 0; $i < 36; $i++)
     {
       sleep(5);
       if (isResRunning("ora.proxy_advm")) { last; } 
     }
   }

   backupOLR();

   backupOCR();

   configureCvuRpm();

   install_xag();

   createMgmtdbDir();

   if (! isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
   {
     gpnp_update_config();
   }

   if ($CFG->platform_family eq "unix")
   {
     createNetLsnrWithUsername(getLsnrUsername());
     if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
     {
       trace("Starting listeners on the node being added ....");
       $success = startListeners(TRUE);
     }
     else
     {
       startNetLsnr(TRUE, getLsnrUsername());
     }
   }

   if ($success) {
      set_bold();
      print_error(325);
      if (1 == $CFG->SUCC_REBOOT)
      {
        print_error(400);
      }
      reset_bold();

      writeCkpt("ROOTCRS_STACK", CKPTSUC);
      
      # Environment variable to be set only on Windows, and only for
      # temporarily supposrting crssetup method for install
      if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # configure remote nodes
      configureRemoteNodes();
      }

   }
   else {
      set_bold();
      print_error(326);
      reset_bold();
      exit 100;
   }
}

sub HAInstall
{
   precheck_siha();

   # instantiate scripts
   instantiate_scripts();

   # Before File Permissions module is invoked, we need to add entries for
   # OCR and Voting Disk locations to crsconfig_fileperms (Bug 8236090).
   add_localOlr_OlrConfig_OcrConfig();

   # run directory creation, files creation/permissions modules
   setup_env();

   if (isCkptexist("ROOTHAS_ACFSINST"))
   {
     trace("Re-run 'acfsroot install' to complete acfs driver install");
     goto ACFS_INST;
   }
    
   # check if CRS is configured
   if (isCRSAlreadyConfigured()) {
      exit 1;
   }

   set_perms_ocr_vdisk();

   osd_setup();

   # prepare to start SIHA
   prepare_to_start_siha();

   # add ons by default for SIHA
   add_ons();

   # start evmd in SIHA mode to replace eonsd functionality
   if (!start_resource("ora.evmd", "-init")) {
      print_error(202);
      exit 1;
   }

   perform_installAFDriver(0);

   # backup OLR
   my $ocrconfig = catfile($CFG->params('ORACLE_HOME'), 'bin', 'ocrconfig');
   my $rc        = system ("$ocrconfig -local -manualbackup");

   if ($rc == 0) {
      trace ("$ocrconfig -local -manualbackup ... passed");
   }
   else {
      trace ("$ocrconfig -local -manualbackup ... failed");
   }

ACFS_INST:
   perform_installUSMDriver(0);

   remove_checkpoints();

   set_bold();
   print_error(327);
   if (1 == $CFG->SUCC_REBOOT)
   {
     print_error(400);
   }
   reset_bold();
}

sub setup_env
{
   # Before Directories Creation module is invoked, we need to add entries
   # for RCALLDIR locations to crsconfig_dirs
   # Note: this is done only on platforms where RCALLDIR is defined.
   if ($CFG->defined_param('RCALLDIR')) {
      add_RCALLDIR_to_dirs ();
   }

   # add ITDI _to crsconfig_dirs
   add_ITDIR_to_dirs();

   # create dirs & set permissions
   my @crsconfig_dirs = read_file(catfile($CFG->ORA_CRS_HOME, 'crs',
                                          'utl', 'crsconfig_dirs'));
   create_dirs (\@crsconfig_dirs);

   copy_wrapper_scripts ();

   my @crsconfig_fileperms = read_file(catfile($CFG->ORA_CRS_HOME, 'crs',
                                               'utl', 'crsconfig_fileperms'));
   set_file_perms (\@crsconfig_fileperms);

   # Set owner/group of ORA_CRS_HOME and its parent dir to root/dba
   if (! is_dev_env() && (! $CFG->SIHA) &&
      ($CFG->platform_family eq "unix"))
   {
      s_setParentDirOwner ($CFG->SUPERUSER, $CFG->ORA_CRS_HOME);
   }

   # create s_crsconfig_$HOST_env.txt file
   s_createConfigEnvFile ();
}

sub add_RCALLDIR_to_dirs
{
    my $SUPERUSER = $CFG->SUPERUSER;
    my $dirsfile  = catfile ($CFG->ORA_CRS_HOME, 'crs', 'utl', 'crsconfig_dirs');

    open (DIRSFILE, ">>$dirsfile") || die(dieformat(208, $dirsfile, $!));

    my $myplatformfamily = s_get_platform_family ();
    $myplatformfamily =~ tr/A-Z/a-z/;

    # add RCALLDIR locations to crsconfig_dirs
    my @RCALLDIRLIST = split (/ /, $CFG->params('RCALLDIR'));
    foreach my $rc (@RCALLDIRLIST) {
        print DIRSFILE "$myplatformfamily $rc $SUPERUSER $SUPERUSER 0755\n";
    }

    close (DIRSFILE);
}

sub add_ITDIR_to_dirs
#---------------------------------------------------------------------
# Function: add IT_DIR directory to crsconfig_dirs
# Args    : none
#---------------------------------------------------------------------
{

   if ($CFG->defined_param('IT_DIR')) {
      if (is_dev_env()) {
         my $itdir     = $CFG->params('IT_DIR');
         my $SUPERUSER = $CFG->SUPERUSER;
         my $dirsfile  = catfile($CFG->ORA_CRS_HOME, 'crs', 'utl', 'crsconfig_dirs');
         my $platform  = s_get_platform_family();

         open (DIRSFILE, ">>$dirsfile")
              or die(dieformat(208, $dirsfile, $!));
         print DIRSFILE "$platform $itdir $SUPERUSER $SUPERUSER 0755\n";
         close (DIRSFILE);
      }
   }
}

sub create_starting_ckpt_entry
{
   # hack to get crs ver from sqlplus. use getcrsrelver once bug 11077430 is fixed
   my $crsrelver = getcrsrelver1();

   if (!isCkptexist("ROOTCRS_STACK")) {
      trace("Oracle clusterware configuration has started");
      writeCkpt("ROOTCRS_STACK", CKPTSTART);
      writeCkptProperty("ROOTCRS_STACK", "VERSION", $crsrelver);
      $CFG->wipCkptName("ROOTCRS_STACK");
   }
   else {
      $CFG->isRerun(TRUE);
      if (!isCkptPropertyExists("ROOTCRS_STACK", "VERSION")) {
         trace("Removing older checkpoints");
         remove_checkpoints();
         trace("Oracle clusterware configuration has started");
         writeCkpt("ROOTCRS_STACK", CKPTSTART);
         writeCkptProperty("ROOTCRS_STACK", "VERSION", $crsrelver);
         $CFG->wipCkptName("ROOTCRS_STACK");
      }
      else {
         my $ckptcrsver = getCkptPropertyValue("ROOTCRS_STACK", "VERSION");
         if (!isVersionMatch($ckptcrsver, $crsrelver)) {
            trace("Removing older checkpoints");
            remove_checkpoints();
            trace("Oracle clusterware configuration has started");
            writeCkpt("ROOTCRS_STACK", CKPTSTART);
            writeCkptProperty("ROOTCRS_STACK", "VERSION", $crsrelver);
            $CFG->wipCkptName("ROOTCRS_STACK");
         }

         my $ckptStatus = getCkptStatus("ROOTCRS_STACK");
         if ($ckptStatus eq CKPTSTART) {
            # It likely that the node crashed.
            # Hence this ckpt is not labeled FAILED
            $CFG->isNodeCrashed(TRUE);
         }

         $CFG->wipCkptName("ROOTCRS_STACK");
      }
   }
}

sub getcrsrelver1
{
   my $home = $_[0];
   my ($sqlplus);
   my $verstring;

   if (! defined $home) {
      $sqlplus = crs_exec_path('sqlplus');
   } else {
      $sqlplus = catfile($home, 'bin', 'sqlplus' );
   }

   my @cmd = ($sqlplus, '-V');
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;
   # if succeeded, parse to ver numbers, output must be a single line,
   # version is 5 numbers, major to minor (see above)
   if ($rc == 0) {
      my ($s1, $s2, $s3, $s4) = split(/ /, $out[1]);
      chomp $s3;
      $verstring = $s3;
      trace( "Got CRS release version: $verstring ");
   }
   else {

      error ("@cmd ... failed rc=$rc with message:\n @out \n");
   }
   return $verstring;
}

sub precheck
{
   if (isPrereqIgnored()) {
      print_error(363);
   }  

   # validate filesystem
   if (! isFilesystemSupported()) {
      writeCkpt("ROOTCRS_STACK", CKPTFAIL);
      exit 1;
   }

   # validate RAC_ON/RAC_OFF
   if (! is_dev_env () && ! isRAC_appropriate()) {
      writeCkpt("ROOTCRS_STACK", CKPTFAIL);
      exit 1;
   }

   my $localNode = $CFG->HOST;
   if ((! isAddNode($localNode, $CFG->params('NODE_NAME_LIST'))) &&
        (! $CFG->UPGRADE))
   {
     # check if we are running first in a rim node
     my $rimNodes = $CFG->params('RIM_NODE_LIST');
     my $profile = catfile($CFG->params('GPNPGCONFIGDIR'), 'gpnp',
                           'profiles', 'peer', 'profile.xml');
     if ($rimNodes =~ /\b$localNode\b/i && !(-e $profile))
     {
       writeCkpt("ROOTCRS_STACK", CKPTFAIL);
       print_error(439);
       exit 1;
     }
     
     # Check if this is not the node where the installer ran.
     # The node where the installer runs must be the first node to configure.
     my $firstNodeToConfig;
     if ((! isFirstNodeToConfig($localNode, $firstNodeToConfig)) &&
          (! (-e $profile)))
     {
       writeCkpt("ROOTCRS_STACK", CKPTFAIL);
       die(dieformat(444, $firstNodeToConfig));
     }
   }
}

sub isFilesystemSupported
{
   if (is_dev_env()) {
      return SUCCESS;
   }

   my $success = SUCCESS;
   my $msg1    = "Root is not able to perform superuser operations " . 
                 "on this filesystem";
   my $msg2    = "Check export options on NAS filer";

   if (! s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                          $CFG->params('ORA_DBA_GROUP'),
                          $CFG->params('ORACLE_HOME')))
   {
      print_error(26, $CFG->params('ORACLE_HOME'));
      trace ("Filesystem of " . $CFG->params('ORACLE_HOME') . 
             " is not supported");
      trace ($msg1);
      trace ($msg2);
      return FAILED;
   }
   else {
      #Reset the ownership of grid home directory to root. Bug 9974887
      s_set_ownergroup($CFG->SUPERUSER,
                       $CFG->params('ORA_DBA_GROUP'),
                       $CFG->params('ORACLE_HOME'));
   }

   if (($CFG->platform_family eq 'unix') &&
       ($CFG->params('CRS_STORAGE_OPTION') == 2))
   {
      my @ocr_locs = split (/\s*,\s*/, $CFG->params('OCR_LOCATIONS'));
      foreach my $loc (@ocr_locs) {
         create_dir (dirname($loc));
         $success = s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                                     $CFG->params('ORA_DBA_GROUP'),
                                     dirname($loc));
         if (! $success) {
            print_error(26,$loc);
            trace ("Filesystem of $loc is not supported");
            trace ($msg1);
            trace ($msg2);
         }
      }
   }

   return $success;
}

sub isInterfaceValid
#-------------------------------------------------------------------------------
# Function:  Check if interface is valid
# Args    :  none
#-------------------------------------------------------------------------------
{
   my $networks       = $CFG->params('NETWORKS');
   my $rc             = FALSE;
   my @interface_list = split (/,/, $networks);
   my $pi_count       = 0;

   foreach my $interface (@interface_list) {
      if ($interface =~ /\bcluster_interconnect\b/) {
         $pi_count++;
      }
   }

   # if more than 1 interface, at least 1 private interface
   # otherwise, it's invalid
   if (scalar(@interface_list) == 1 ||
      (scalar(@interface_list) > 1 && $pi_count >= 1)) {
      $rc = TRUE;
   }
   else {
      print_error(354);
      trace ("There are more than one interface,");
      trace ("but there is no private interface specified");
   }

   return $rc;
}

sub save_param_file
{
   # save param file
   my $ckptstatus = getCkptStatus("ROOTCRS_STACK");

   trace("Saving the configuration parameter file data");
   trace("checkpoint status of ROOTCRS_STACK is $ckptstatus");

   if (!isCkptexist("ROOTCRS_PARAM") && ($ckptstatus ne CKPTSUC)) {
      trace("inside rootcrs_param");
      writeCkpt("ROOTCRS_PARAM", CKPTSTART);
      writeCkptPropertyFile("ROOTCRS_PARAM", $CFG->paramfile);
      writeCkpt("ROOTCRS_PARAM", CKPTSUC);
   }
}

sub osd_setup
{
   if (!isCkptexist("ROOTCRS_OSDSETUP")) {
      #This checkpoint is mainly required on windows
      trace("Writing a checkpoint for platform specific OSD setup");
      writeCkpt("ROOTCRS_OSDSETUP", CKPTSTART);
      $CFG->wipCkptName("ROOTCRS_OSDSETUP");
   }

   if (!isCkptSuccess("ROOTCRS_OSDSETUP")) {
      $CFG->wipCkptName("ROOTCRS_OSDSETUP");

      # perform platform-specific setup actions
      # only for fresh install
      if (SUCCESS != s_osd_setup()) {
         print_error(193);
         writeCkpt("ROOTCRS_OSDSETUP", CKPTFAIL);
         exit 1;
      }
      else {
         trace("Platform specific setup actions are done");
         writeCkpt("ROOTCRS_OSDSETUP", CKPTSUC);
         $CFG->wipCkptName("ROOTCRS_STACK");
      }
   }
}

sub prepare_to_start_clusterware
{
   # Check if CRS is already configured
   my $crsConfigured   = FALSE;
   my $gpnp_setup_type = GPNP_SETUP_BAD;

   check_CRSConfig($CFG->ORA_CRS_HOME,
                   $CFG->HOST,
                   $CFG->params('ORACLE_OWNER'),
                   $CFG->params('ORA_DBA_GROUP'),
                   $CFG->params('GPNPGCONFIGDIR'),
                   $CFG->params('GPNPCONFIGDIR'),
                   $crsConfigured,
                   $gpnp_setup_type) || die(dieformat(315));

   if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
   {
     # For fresh insall, the local gpnp profile is a copy of stage profile at this point,
     # so it doesn't make sense to update the gpnp stage profile right away.
     trace("Updating the clusterwide gpnp stage profile with the latest node profile");
     gpnp_update_config();
   }

   # Verify bdb location. Initially its in CRS HOME. oclumon changes later
   if (isCRFSupported() && !is_dev_env() && !isReposBDB()) {
      my $bdbloc = catfile($CFG->ORA_CRS_HOME, 'crf', 'db', $CFG->HOST);
      s_crf_check_bdbloc($bdbloc, $CFG->params('NODE_NAME_LIST'));
   }

   if (!isCkptSuccess("ROOTCRS_STACK")) {
      #set $crscConfigured to false for checkpoint logic
      $crsConfigured = FALSE;
   }

   # return if it's already configured
   if ($crsConfigured) {
      return SUCCESS;
   }

   # Check existing configuration, Create and populate OLR and OCR keys
   perform_olr_config() || die(dieformat(316));

   # Initialize the SCR settings.
   s_init_scr ();

   # create GPnP wallet/profile
   initialize_local_gpnp($CFG->HOST, $gpnp_setup_type);

   create_IPDOS_config();

   # Setup OHASD service/daemon
   perform_register_ohasd() || die(dieformat(317));

   # Start OHASD service/daemon
   perform_start_ohasd() || die(dieformat(318));

   # create OHASD resources
   create_ohasd_resources();

   # Install Drivers - USM, OKA, AFD
   perform_installDrivers();
   # bounce the ohasd so that the ohasd resources/types take effect
   stopFullStack("force") || die(dieformat(349));

   # Do initial cluster configuration. 
   # It handles first and non-first nodes appropriately
   if (! perform_init_config())
   {
      print_error(198);
      exit 1;
   }

   # Start full stack
   start_cluster();
}

sub create_asm_credentials
#-------------------------------------------------------------------------------
# Function: Create and import ASM credentials. 
#           This is called in a Local and Near ASM clusters.
# Args    : None.
# Returns : None.
#-------------------------------------------------------------------------------
{
  if (isFirstNodeToStart())
  {
    # the OCR domain is created once on the first node
    if (FAILED == createCredDomain('ASM', 'OCR'))
    {
      die(dieformat(377));
    }

    if (isNearASM())
    {
      if (FAILED == createASMCredentials())
      {
        die(dieformat(365));
      }
      import_asm_credentials();
      push_seed_dir($CFG->params('NODE_NAME_LIST'));
    }
  }
}


sub copy_asm_credentials
#-------------------------------------------------------------------------------
# Function: Copy and import ASM credentials. 
#           This is called on the first node of a ASM client cluster.
# Args    : None.
# Returns : None.
#-------------------------------------------------------------------------------
{
  if (FAILED == createCredDomain('ASM', 'OCR'))
  {
    die(dieformat(377));
  }

  if (FAILED == copyASMCredentials())
  {
    die(dieformat(375));
  }

  import_asm_credentials();
  push_seed_dir($CFG->params('NODE_NAME_LIST'));
}

sub import_asm_credentials
#-------------------------------------------------------------------------------
# Function: Import ASM credentials into OLR.
#           This is called on Local, Near and Far ASM clusters.
# Args    : None.
# Returns : None.
#-------------------------------------------------------------------------------
{
  if (FAILED == createCredDomain('ASM', 'OLR') )
  {
    die(dieformat(377));
  }

  if (FAILED == importASMCredentials())
  {
    die(dieformat(366));
  }
}

sub start_cluster
{
  trace(sprintf("Startup level is %d", $CFG->stackStartLevel));

  # start the entire stack in shiphome
  if (START_STACK_ALL == $CFG->stackStartLevel)
  {
    trace("Attempt to start the whole CRS stack");
    my $rc = startFullStack($CFG->params('ORACLE_HOME'));
    if ( FAILED == $rc)
    { 
      die(dieformat(117));
    }
    elsif (WARNING == $rc)
    {
      # maximum number of hub nodes reached, try this as a rim node.
      my $role = NODE_ROLE_RIM;
      setNodeRole($role);
      stopFullStack("force") || die(dieformat(349));
      $rc = startFullStack($CFG->params('ORACLE_HOME'), $role);
      if (SUCCESS != $rc)
      { 
        die(dieformat(117));
      }
      else
      {
        print_error(343);
        return;
      }
    }
    else
    {
      print_error(343);
      return;
    }
  }

  # Start OHASD right here so that CRS stack can get started up
  # to the specified level
  startOhasdOnly() || die(dieformat(117));

  # Start EVM
  if ($CFG->stackStartLevel >= START_STACK_EVMD)
  { 
    if ( ! start_resource("ora.evmd", "-init") )
    {
      print_error(117);
      die(dieformat(250));
    }
  }

  # Start mdnsd
  if ($CFG->stackStartLevel >= START_STACK_MDNSD)
  {
    if (!start_resource("ora.mdnsd", "-init"))
    {
      print_error(117);
      # TODO: add an entry for mdnsd start failure to clsrscus.msg
      die(dieformat(117));
    }
  }

  # Start gpnpd
  if ($CFG->stackStartLevel >= START_STACK_GPNPD)
  {
    if (!(start_resource("ora.gpnpd", "-init") && wait_for_gpnpd_start()))
    {
      print_error(117);
      die(dieformat(242));
    }
  }

  # Start gipcd
  if ($CFG->stackStartLevel >= START_STACK_GIPCD)
  {
    if (!(start_resource("ora.gipcd", "-init")))
    {
      print_error(117);
      # TODO: add an entry for gipcd start failure to clsrscus.msg
      die(dieformat(117));
    }
  }

  # Start CSS in clustered mode
  if ($CFG->stackStartLevel >= START_STACK_CSSD)
  {
    if ( ! CSS_start_clustered() )
    {
      print_error(117);
      die(dieformat(244));
    }
  }

  # Start CTSS with reboot option to signal step sync
  # Note: Before migrating stack startup to 'crsctl start crs',
  #       'CTSS_REBOOT=TRUE' is a workaround to signal step sync.
  if ($CFG->stackStartLevel >= START_STACK_CTSSD)
  {
    if ( ! start_resource("ora.ctssd", "-init",
                            "-env", "USR_ORA_ENV=CTSS_REBOOT=TRUE") )
    {
      print_error(117);
      die(dieformat(245));
    }
  }

  # Start CRF
  if ($CFG->stackStartLevel >= START_STACK_CRF)
  {
    if (! isCRFSupported() || ! start_resource("ora.crf", "-init"))
    {
      trace ("start_cluster: Failed to start CRF");
      # We don't want IPD failure to stop rest of the stack
      # from coming up. So, no exit here!
    }
  }

  # startup the HAIP if it has been configured
  if ($CFG->stackStartLevel >= START_STACK_HAIP)
  {
    enable_HAIP();
  }

  # Start ASM if needed.
  if ($CFG->stackStartLevel >= START_STACK_ASM)
  { 
    if (($CFG->params('CRS_STORAGE_OPTION') == 1))
    {
      my $rc;

      if (isLegacyASM())
      {
        trace("Starting ASM on all nodes for Local ASM");
        $rc = start_resource("ora.asm", "-init");
      }
      elsif (isNearASM() && isFirstNodeToStart())
      {
        trace("Starting ASM on the first node for Near ASM");
        $rc = start_resource("ora.asm", "-init");
      }
      else
      {
        trace("No need to start ASM on this node");
        $rc = SUCCESS;
      }

      if ($rc != SUCCESS)
      {
        print_error(117);
        die(dieformat(247));
      }
    }
  }

  # Start CRS
  if ($CFG->stackStartLevel >= START_STACK_CRSD)
  { 
    if ( ! start_resource("ora.crsd", "-init") )
    {
      print_error(117);
      die(dieformat(249));
    }
  }

  if ($CFG->stackStartLevel >= START_STACK_CRSD)
  {
    if (!wait_for_stack_start(360))
    {
      die(dieformat(251));
    }
  }
  else
  {
    # Stack started up partially.
    trace("Started the Clusterware stack to level $CFG->stackStartLevel");
  }

  print_error(343);
}

sub create_IPDOS_config
{
   # Create CHM/OS config
   my $bdbloc;
   my $crfconfig    = catfile($CFG->ORA_CRS_HOME, 'crf', 'admin',
                              'crf' . $CFG->HOST . '.ora');
   my $tmpcrfconfig = crf_config_generate($CFG->HOST,
                                          "default",
                                          $CFG->params('ORACLE_OWNER'),
                                          $CFG->params('NODE_NAME_LIST'));

   trace ("Creating CHM/OS config file $crfconfig");

   #  Delete older existing CHM/OS repository for Upgrade but 
   #    for new installation delete the newer location.
   # If not removed, leads to CHM/OS repository corruption.
   if ($CFG->UPGRADE)
   {
     my $old_crshome = $CFG->OLD_CRS_HOME;
     my $oldcrfconfig = catfile($old_crshome, 'crf', 'admin',
				'crf' . $CFG->HOST . '.ora');
     $bdbloc = getCHMAttrib("BDBLOC", $oldcrfconfig);
     if ($bdbloc eq "default")
     {
       $bdbloc = catfile($old_crshome, 'crf', 'db', $CFG->HOST);
     } 
   }
   
   my $new_crs_running = FALSE;
   # Check if New Grid Infrastructure is running
   if ($CFG->isRerun)
   {
     trace("Trying to check if new CRS stack is partially up");
     $new_crs_running = check_NewCrsStack();
   }   

   # Dont delete the bdb only if the new stack is running.
   if (!$new_crs_running)
   {
     trace ("Deleting older CHM/OS repository at $bdbloc");
     crf_delete_bdb($bdbloc);
   }

   copy_file($tmpcrfconfig, $crfconfig);
   unlink $tmpcrfconfig;

   # delete older IPD/OS installation
   crf_do_delete();
}

sub create_ohasd_resources
{
   ###FIXME: Change once we know types of resources to be created
   trace ("Creating OHASD resources and dependencies");

   my $status;
   my $hasdconfig_superuser = $CFG->SUPERUSER;
   if ($OSNAME eq 'MSWin32' && is_dev_env()) {
      $hasdconfig_superuser = $CFG->params('ORACLE_OWNER');
   }

   trace("Configuring HASD");
   if ($CFG->platform_family eq "windows") {
      my $NT_AUTHORITY = '';
      if (is_dev_env()) {
         $NT_AUTHORITY = $CFG->params('ORACLE_OWNER');
      }

      $status = perform_configure_hasd('crs',
                                       $CFG->HOST,
                                       $NT_AUTHORITY,
                                       $NT_AUTHORITY,
                                       $CFG->params('ORA_DBA_GROUP'));
   }
   else {
      $status = perform_configure_hasd('crs',
                                       $CFG->HOST,
                                       $CFG->params('ORACLE_OWNER'),
                                       $hasdconfig_superuser,
                                       $CFG->params('ORA_DBA_GROUP'));
   }

   if ($status) {
      trace ("Successfully created OHASD resources for cluster and ASM");
   }
   else {
      print_error(195);
   }
}

# This code installs USM, OKA and Asm Filter Drivers.
sub perform_installDrivers
{
   #
   # IMPORTANT NOTE:  ****
   #           perform_installUSMDriver() & perform_installKADriver()
   #           and perform_installAFDriver
   #           have to be kept together so that the user only needs to 
   #           reboot 1 time at max. (if driver unload/load fails).
   #

   # install ASM Filter Driver
   perform_installAFDriver(1); 

   # install USM driver
   perform_installUSMDriver(1);

   # install KA driver. Supported only in cluster.
   #perform_installKADriver();

}

#   This code contains the functionality to install the USM
# driver by calling code in the oraacfs.pm module.
# 
# ARGS:
#  $chkpoints - 0 or 1 In SIHA, there is no chkpoint functionality.
#      0 - Don't use checkpoints
#      1 - Go ahead and use checkpoints.
sub perform_installUSMDriver
{
   my $chkpoints = $_[0];
   my $has = "has";

   if ($chkpoints == 1)
   {
     $has = "crs";
   }

   if ($chkpoints == 1)
   {
       if (!isCkptexist("ROOTCRS_ACFSINST")) {
          trace("Writing checkpoint for USM driver install");
          writeCkpt("ROOTCRS_ACFSINST", CKPTSTART);
          $CFG->wipCkptName("ROOTCRS_ACFSINST");
       }
   }
   if (($chkpoints != 1) ||  (!isCkptSuccess("ROOTCRS_ACFSINST")) ){
      $CFG->wipCkptName("ROOTCRS_ACFSINST");

      if (1 == $chkpoints)
      {
        # Only ohasd needs to be up before installing acfs drivers
        bounce_ohasd();
      }
   
      # install acfsroot kernel
      my $ret = installUSMDriver($has);
      if (FAILED == $ret)
      {
         # This is some failure that doesn't relate to driver load\unload.
         # For instance, this could be an issue generating symbols.
         print_error(196);
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_ACFSINST", CKPTFAIL);
         }
         exit 1;
      }
      elsif (REBOOT == $ret)
      {
         # We couldn't unload the old drivers or load the new ones.
         # CRS is now disabled, so the user can reboot and try again.
         trace("ACFS drivers unable to be installed.");
   	 set_bold();
         print_error(400);
   	 reset_bold();
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_ACFSINST", CKPTFAIL);
         }
         else
         {
           writeCkpt("ROOTHAS_ACFSINST", CKPTFAIL);
         }
         exit 1;
      }
      else
      {
         # It worked!
         trace("ACFS drivers installation completed");
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_ACFSINST", CKPTSUC);
           $CFG->wipCkptName("ROOTCRS_STACK");
         }
      }
   }
}

sub perform_installAFDriver
{
   my $chkpoints = $_[0];

   # HA does not honour checkpoint functionality
   if ($chkpoints == 1)
   {
      if (!isCkptexist("ROOTCRS_AFDINST")) {
       trace("Writing checkpoint for AFD driver install");
       writeCkpt("ROOTCRS_AFDINST", CKPTSTART);
       $CFG->wipCkptName("ROOTCRS_AFDINST");
      }
   }

   if (($chkpoints != 1) || !isCkptSuccess("ROOTCRS_AFDINST")){
      $CFG->wipCkptName("ROOTCRS_AFDINST");

      # install afd driver
      my $ret = installAFDriver();
      if (FAILED == $ret)
      {
         # This is some failure that doesn't relate to driver load\unload.
         # For instance, this could be an issue generating symbols.
         print_error(2501);
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_AFDINST", CKPTFAIL);
         }
         exit 1;
      }
      elsif (REBOOT == $ret)
      {
         trace("ASM filter drivers unable to be installed.");
   	 set_bold();
         print_error(400);
   	 reset_bold();
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_AFDINST", CKPTFAIL);
         }
         exit 1;
      }
      else
      {
         # It worked!
         trace("AFD drivers installation completed");
         if ($chkpoints == 1)
         {
           writeCkpt("ROOTCRS_AFDINST", CKPTSUC);
           $CFG->wipCkptName("ROOTCRS_STACK");
         }
      }
   }
}


# This code contains the functionality to install the KA
# driver by calling code in the crska.pm module.
# 
sub perform_installKADriver
{
   my $has = "crs";

   #
   # Unconditionally perform_installKADriver() is called
   # by the caller. Hence, we need to check if OKA
   # is supported before proceeding.
   #
   if (!isOKASupported())
   {
      # If OKA not supported,
      # then assume everything is okay.
      trace("OKA is not supported on this platform.");
      return;
   }

   if (!isCkptexist("ROOTCRS_OKAINST")) {
      trace("Writing checkpoint for OKA driver install");
      writeCkpt("ROOTCRS_OKAINST", CKPTSTART);
      $CFG->wipCkptName("ROOTCRS_OKAINST");
   }

   if (!isCkptSuccess("ROOTCRS_OKAINST")){
      $CFG->wipCkptName("ROOTCRS_OKAINST");
   
      # install okaroot kernel
      my $ret = installOKADriver($has);
      if (FAILED == $ret)
      {
         # This is some failure that doesn't relate to driver load\unload.
         # For instance, this could be an issue generating symbols.
         print_error(2007);
         writeCkpt("ROOTCRS_OKAINST", CKPTFAIL);
         exit 1;
      }
      elsif (REBOOT == $ret)
      {
         # We couldn't unload the old drivers or load the new ones.
         # CRS is now disabled, so the user can reboot and try again.
         trace("OKA drivers unable to be installed.");
   	 set_bold();
         print_error(400);
   	 reset_bold();
         writeCkpt("ROOTCRS_OKAINST", CKPTFAIL);
         exit 1;
      }
      elsif (WARNING == $ret)
      {
         #
         # In case of driver install/load failure,
         # we just proceed with out enabling the
         # ora.drivers.oka resource.
         #
         # Warning !
         trace("OKA install failed. Continuing with rest of the installation.");

         writeCkpt("ROOTCRS_OKAINST", CKPTSUC);
         $CFG->wipCkptName("ROOTCRS_STACK");
      }
      else
      {
         # It worked!
         trace("OKA drivers installation completed");

         # Enable the ora.drivers.oka resource.
         my $crsctl = crs_exec_path('crsctl');
         my @mycmd = ($crsctl, 'modify', 'res', 'ora.drivers.oka', '-attr',
                       "\"ENABLED=1\"", '-init');
         my $myrc = system_cmd(@mycmd);
         if ($myrc != 0)
         {
            trace("Enabling of ora.drivers.oka resource failed. Continuing with rest of the installation.");
         }

         writeCkpt("ROOTCRS_OKAINST", CKPTSUC);
         $CFG->wipCkptName("ROOTCRS_STACK");
      }
   }
}

####---------------------------------------------------------
#### Function for checking if CRS is already configured
# ARGS : 1
# ARG1 : CRS home
# ARG2 : Host name
# ARG3 : CRS user
# ARG4 : isCrsConfigured? (OUT var)
sub check_CRSConfig
{
    my $crshome   = $_[0];
    my $hostname  = $_[1];
    my $crsuser   = $_[2];
    my $dbagroup  = $_[3];
    my $gpnpghome = $_[4];
    my $gpnplhome = $_[5];
    my $crsconfigok = FALSE;
    my $gpnp_setup_type = GPNP_SETUP_BAD;

    # init outers
    $_[6] = $crsconfigok;
    $_[7] = $gpnp_setup_type;

    trace ("Oracle CRS home = " . $crshome);

    if (!$hostname) {
        print_error(15);
        return FAILED;
    }

    trace ("Host name = " . $hostname);

    if (!$crsuser) {
        print_error(16);
        return FAILED;
    }

    trace ("CRS user = " . $crsuser);

    ## Define gpnp globals and validate gpnp directories.
    # Note: This step must be performed unconditionally,
    #       because successfull script use gpnp globals.
    #
    if (! verify_gpnp_dirs( $crshome, $gpnpghome, $gpnplhome,
                            $hostname, $crsuser, $dbagroup ) ) {
        trace("GPnP cluster-wide dir: $gpnpghome, local dir: $gpnplhome.");
        print_error(17);
        return FAILED;
    }

    if ($CFG->JOIN)
    {
      if (! pull_cluserwide_gpnp_setup($CFG->EXISTINGNODE))
      {
        my $nodeName = $CFG->EXISTINGNODE;
        die(dieformat(440, $nodeName));
      }
    }

    if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
    {
      trace("Copy GPnP setup and ASM credentials from existing nodes ".
            "if they are not present");

      my $succ = SUCCESS;
      if (! check_clusterwide_gpnp_profile())
      {
        $succ = pull_cluserwide_gpnp_setup2($CFG->params('NODE_NAME_LIST'));
      }

      if ($succ)
      {
        $succ = pull_asm_cred_file($CFG->params('NODE_NAME_LIST'));
      }
      
      if (SUCCESS != $succ)
      {
        die(dieformat(442));
      }
    }

    ##Checking if CRS has already been configured
    trace ("Checking to see if Oracle CRS stack is already configured");

    # call OSD API
    if (s_check_CRSConfig ($hostname, $crsuser)) {
        $crsconfigok = TRUE;
    }

    ## GPnP validate existing setup, if any
    #  If a cluster-wide setup found, it will be promoted to local
    $gpnp_setup_type = check_gpnp_setup( $crshome,
                                         $gpnpghome, $gpnplhome, $hostname,
                                         $crsuser, $dbagroup );

    if ($gpnp_setup_type != GPNP_SETUP_GOTCLUSTERWIDE &&
        $gpnp_setup_type != GPNP_SETUP_CLUSTERWIDE)
    {
      trace ("GPNP configuration required");
      $crsconfigok = FALSE;  # gpnp setup is not ok or not finalized
    }
    $CFG->gpnp_setup_type($gpnp_setup_type);

    # reinit outers
    $_[6] = $crsconfigok;
    $_[7] = $gpnp_setup_type;
   return SUCCESS;
}

sub perform_olr_config
{
   my $ckptName = "ROOTCRS_OLR";
   if (isOLRConfigured()) {
      return SUCCESS;
   }

   my $OCR = $CFG->params('OCR_LOCATIONS');
   if ($CFG->ASM_STORAGE_USED) {
      $OCR = "+" . $CFG->params('ASM_DISK_GROUP');
   }

   validateOCR($CFG->ORA_CRS_HOME,
               $CFG->params('CLUSTER_NAME'),
               $OCR) || die(dieformat(293, $OCR));

   # delete olr file if exists
   if (-e $CFG->OLR_LOCATION) {
      if ($CFG->DEBUG) { trace ("unlink " . $CFG->OLR_LOCATION); }
      unlink ($CFG->OLR_LOCATION) || print_error(102, $CFG->OLR_LOCATION, $!);
   }

   if (! $CFG->SIHA) {
      initial_cluster_validation();
   }

   upgrade_local_OCR();

   create_OLR_keys();

   writeCkpt($ckptName, CKPTSUC);
   print("OLR initialization - successful\n");
   return SUCCESS;
}

sub isOLRConfigured 
{
   my $ckptName = "ROOTCRS_OLR";

   if (isCkptexist($ckptName)) {
      my $ckptStatus = getCkptStatus($ckptName);
      trace("'$ckptName' state is '$ckptStatus'");

      if (isCkptSuccess($ckptName)) {
         trace("OLR is already initialized");
         $CFG->wipCkptName("ROOTCRS_STACK");
         return TRUE;
      }
   }

   trace("Initializing OLR now..");
   writeCkpt($ckptName, CKPTSTART);
   $CFG->wipCkptName($ckptName);
   return FALSE;
}

=head2 initial_cluster_validation

  Perform validations for the cluster installation as well as
  initializes some component files

=head3 Parameters

  None

=head3 Returns

  None, errors result in termination of the script

=cut

sub initial_cluster_validation
{
   validate_SICSS() || die(dieformat(290));
   validate_olrconfig($CFG->OLR_LOCATION,
                      $CFG->ORA_CRS_HOME) || die(dieformat(292));

   if (!CSS_CanRunRealtime($CFG)) {
      die(dieformat(294));
   }

   if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST'))) {
      if ((isOCRonASM()) && ($CFG->params('CRS_STORAGE_OPTION') != 1)) {
         $CFG->params('CRS_STORAGE_OPTION', 1);
      }
      elsif ((! isOCRonASM()) &&
             ($CFG->params('CRS_STORAGE_OPTION') == 1)) {
             $CFG->params('CRS_STORAGE_OPTION', 2);
      }
   }
}

# delete a node from the CRF install
sub crf_do_delete
{
   # shutdown the sysmond, ologgerd, oproxyd if they are running
   my ($cmd, $instdir, $defpath, $rootpath, $configfile, $line, $bdbloc,
       $admindir, $admin, $runpth);

   trace("Check and delete older IPD/OS installation");
   if ($CFG->platform_family eq "windows") {
      $instdir  = "C:\\Program Files\\oracrf";
      $defpath  = "$ENV{SYSTEMROOT}"."\\system32\\";
      $rootpath = "$ENV{SYSTEMROOT}";
   }
   else {
      $instdir  = "/usr/lib/oracrf";
      $defpath  = "/usr/bin/";
      $rootpath = "/";
   }

   my $instfile = catfile($instdir, "install", "installed");

   if (! -f $instfile) {
      trace("INFO: The OS Tool is not installed at $instdir.");
   }
   else {
      trace("Older IPD/OS installation detected ... Stopping and removing it...");
      $cmd = "oclumon"." ". "stop"." ". "all";
      my $crfhome = $instdir;
      $admindir = catfile("$instdir", "crf");
      if ( -d $admindir) {
         $configfile = catfile($instdir, "crf", "admin", 
                               "crf" . $CFG->HOST . ".ora");
         $runpth     = catfile($crfhome, "crf", "admin", "run");
         $admin      = 'crf';
      }
      else {
         $configfile = catfile($instdir, "admin", "crf" . $CFG->HOST . ".ora");
         $runpth     = catfile($crfhome, "admin", "run");
         $admin      = 'admin';
      }

      system("$cmd");
      sleep(5);

      # read config file to find older BDB loc
      $bdbloc = getCHMAttrib("BDBLOC", $configfile);
      if ($bdbloc eq "default")
      {
        $bdbloc = catfile($instdir, "crf", "db", "$CFG->HOST");
      }

      my $pidf=catfile($runpth, "crfmond", "s" . $CFG->HOST . ".pid");
      if (-f $pidf) {
         open(PID_FILE, $pidf);
         while (<PID_FILE>) {
            crf_kill_for_sure($_);
         }
         close(PID_FILE);
         unlink($pidf);
      }

      my $dir = catfile("$runpth", "crfmond");
      rmdir("$dir");

      # ologgerd now
      $pidf=catfile($runpth, "crflogd", "l" . $CFG->HOST . ".pid");
      if (-f $pidf) {
         open(PID_FILE, $pidf);
         while (<PID_FILE>) {
            crf_kill_for_sure($_);
         }

         close(PID_FILE);
         unlink($pidf);
      }

      $dir = catfile($runpth, "crflogd");
      rmdir("$dir");

      # proxy next
      $pidf=catfile($runpth, "crfproxy", "p" . $CFG->HOST . ".pid");
      if (-f $pidf) {
         open(PID_FILE, $pidf);
         while (<PID_FILE>) {
            crf_kill_for_sure($_);
            # give some time to oproxy to react.
            sleep 2;
         }

         close(PID_FILE);
         unlink($pidf);
      }

      $dir = catfile($runpth, "crfproxy");
      rmdir("$dir");

      # ask crfcheck to shutdown cleanly
      $pidf=catfile($crfhome, "log", $CFG->HOST, "crfcheck", "crfcheck.lck");
      if (-f $pidf) {
         open(PID_FILE, $pidf);
         while (<PID_FILE>) {
            kill 15, $_;
         }
         close(PID_FILE);
      }

      my $rootpath;
      s_crf_remove_itab();

      # remove the tree
      trace("Removing install dirs from $instdir ...\n");
      my $filed;
      my $file;
      foreach $filed ('bin', 'lib', $admin, 'jlib', 'mesg', 'log',
                     'install', 'jdk', 'db')
      {
         $file = catfile($instdir, "$filed");
         rmtree("$file", 0, 0);
      }

      # delete old bdb files.
      trace("Deleting older IPD/OS BDB files at: ", $bdbloc);
      crf_delete_bdb($bdbloc);

      unlink("$defpath"."crfgui");
      unlink("$defpath"."oclumon");
      unlink("$defpath"."ologdbg");

      # change dir to a safer place
      chdir $rootpath;
      trace("Removing CRFHOME path $instdir...\n");
      rmdir $instdir;
      trace("Old IPD/OS install removal operation completed.\n");
   }
}

sub crf_kill_for_sure
{
   kill(15, $_[0]);

   # if that didn't work, use force
   if (kill(0, $_[0])) {
      kill(9, $_[0]);
   }
}

sub add_clscfg
{
   # add clscfg for new node
   if (!isCkptexist("ROOTCRS_ADDNODE")) {
      trace("Writing checkpoint for add node actions");
      writeCkpt("ROOTCRS_ADDNODE", CKPTSTART);
      $CFG->wipCkptName("ROOTCRS_ADDNODE");
   }

   if (!isCkptSuccess("ROOTCRS_ADDNODE")) {
      $CFG->wipCkptName("ROOTCRS_ADDNODE");
      writeCkpt("ROOTCRS_ADDNODE", CKPTSTART);
      my $status = run_crs_cmd('clscfg', '-add');

      if ($status == 0) {
         writeCkpt("ROOTCRS_ADDNODE", CKPTSUC);
         trace ("clscfg -add completed successfully");
      }
      else {
         writeCkpt("ROOTCRS_ADDNODE", CKPTFAIL);
         print_error(180, "clscfg -add", $status);
         exit 1;
      }
   }
}

# checkpoint wrapper function for perform_initial_config
sub perform_init_config
{
   my $ckptStatus;
   my $ckptName = "ROOTCRS_BOOTCFG";

   if (isCkptexist($ckptName)) {
      $ckptStatus = getCkptStatus($ckptName);
      trace("'$ckptName' state is $ckptStatus");

      if ((($ckptStatus eq CKPTSTART) && $CFG->isNodeCrashed) || 
           ($ckptStatus eq CKPTFAIL))
      {
         clean_perform_initial_config();
         $CFG->isNodeCrashed(FALSE);
      } 
      elsif ($ckptStatus eq CKPTSUC) {
         trace("Node specific initial boot configuration already completed");
         $CFG->wipCkptName("ROOTCRS_STACK");
         return SUCCESS;
      }
   }

   return perform_initial_config();
}

sub clean_perform_initial_config
{
  trace("Clean perform_initial_config");
  stopFullStack("force") || die(dieformat(349));
}

=head2 perform_initial_config

   Checks for existing CSS configuration and creates initial
   configuration if no configuration found

=head3 Parameters

   The parameter hash

=head3 Returns

  TRUE  - A  CSS configuration was found or created
  FALSE - No CSS configuration was found and none created

=cut

sub perform_initial_config {
  my $rc;
  my $success = TRUE;
  my $ASM_DISK_GROUP = $CFG->params('ASM_DISK_GROUP');
  my $ckptName = "ROOTCRS_BOOTCFG";
  my $excl_ret;
  my $css_up_normal = FALSE;
  my $localNode = $CFG->HOST;

  ## Enter exclusive mode to setup the environment
  trace ("Checking if initial configuration has been performed");

  writeCkpt($ckptName, CKPTSTART);
  $CFG->wipCkptName($ckptName);

  my $firstNodeToConfig;
  if (! isFirstNodeToConfig($localNode, $firstNodeToConfig))
  {
    trace("Node $localNode is not the first node to configure, ".
          "hence do not start CSS in exclusive mode");

    startOhasdOnly() || die(dieformat(117));

    # Import credentials for ASM on non-first nodes, including rim nodes
    # A rim node can never be a first node, hence CSS is never started in X mode
    # on a rim node.

    # Get the ASM mode from the gpnp global stage profile for addnode case
    my ($rc, $asmmode);
    if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST')))
    { 
      ($rc, $asmmode) = gpnp_get_asm_mode(get_peer_profile_file(FALSE));
    }
    trace("ASM mode = $asmmode");

    # The ASM mode doesn't get updated in the local gpnp profile
    # until the remote ASM is enabled.
    if ((! isLegacyASM()) || (ASM_MODE_FLEX eq $asmmode))
    {
      trace("Importing asm credentials");
      import_asm_credentials();
    }

    stopFullStack("force") || die(dieformat(349));

    writeCkpt($ckptName, CKPTSUC);
    $CFG->wipCkptName("ROOTCRS_STACK");

    return $success;
  }

  if (checkServiceDown("css")) {
    $excl_ret = CSS_start_exclusive();
  } elsif (!CSS_is_configured()) {
    $excl_ret = CSS_EXCL_SUCCESS;
  } else {$css_up_normal = TRUE;}


  if ((!$css_up_normal) && $excl_ret != CSS_EXCL_SUCCESS) 
  {
    if ($excl_ret == CSS_EXCL_FAIL_CLUSTER_ACTIVE) 
    {
      # The exclusive mode startup failure should not happen due to
      # another node being already up
      print_error(443);
      $success = FALSE;
    }
    else {
      trace("The exlusive mode cluster start failed, see Clusterware alert log",
            "for more information");
      print_error(119);
      $success = FALSE;
    }

    stopFullStack("force");
  }

  # in business
  # Need to find out whether we should be doing something as
  # exclusive node or not. Use CSS voting files as a way of
  # checking cluster initialization status
  elsif (CSS_is_configured()) {
    trace("Existing configuration setup found");
  }
  else {
    trace("This is the first node to start");
    trace("Performing initial configuration for cluster");

    if (isFarASM())
    {
      trace("Copy and import ASM credentials on the first node of a ASM client cluster");
      copy_asm_credentials();
    }

    if (!start_resource("ora.ctssd", "-init")) 
    {
      $success = FALSE;
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(257));

    }
    # If ASM diskgroup is defined, need to configure and start ASM
    # ASM is started as part of the config
    elsif (!isFarASM() && $CFG->ASM_STORAGE_USED)
    {
      if (isODA())
      {
        $success = configure_ASM_oda();
      }
      else
      {
        $success = configure_ASM();
      }
      if (!$success)
      {
        writeCkpt($ckptName, CKPTFAIL);
        die(dieformat(258));
      }
    }
    # ocrconfig - Create OCR keys
    if (! configure_OCR()) {
      $success = FALSE;
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(259));
    }
    elsif (!startExclCRS()) {
      $success = FALSE;
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(260));
    } 
    elsif (isFarASM() && FAILED == createCredDomain('ASM', 'OCR'))
    {
      $success = FALSE;
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(377));
    }
    else 
    {
      trace ("Creating voting files");
      if (!add_voting_disks()) {
          writeCkpt($ckptName, CKPTFAIL);
          die(dieformat(261));
      }

      my $CRSCTL = crs_exec_path('crsctl'); 
      system("$CRSCTL query css votedisk");
    }

    trace("The node:" . $CFG->HOST . " is the first node to configure");

    if (($success) && (isBigCluster()) &&
         (! ($CFG->params('HUB_NODE_LIST'))) &&
         ($CFG->defined_param('HUB_SIZE')))
    {
      $success = setHubSize();
    } 
 
    # Allow additional commands to be executed if set
    if ($success && $CFG->CRSCFG_POST_CMD) {
      my @cmdl = @{$CFG->CRSCFG_POST_CMD};
      for my $cmd (@cmdl) { my @cmd = @{$cmd}; system_cmd(@cmd); }
    }
  }

  if ($excl_ret == CSS_EXCL_SUCCESS)
  {
    # Push local gpnp setup to be cluster-wide.
    # This will copy local gpnp file profile/wallet setup to a
    # list of cluster nodes, including current node.
    # This promotes a node-local gpnp setup to be "cluster-wide"
    trace ("Promoting local gpnp setup to cluster-wide. Nodes " .
           $CFG->params('NODE_NAME_LIST'));

    if (! push_clusterwide_gpnp_setup( $CFG->params('NODE_NAME_LIST')))
    {
      $success = FALSE;
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(262));
    }

    trace("Stopping CSS which is running exclusive mode");
    if (! stopFullStack("force"))
    {
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(267));
    }
  }

  if ($success != TRUE) {
     writeCkpt($ckptName, CKPTFAIL);
  } else {
     writeCkpt($ckptName, CKPTSUC);
     $CFG->wipCkptName("ROOTCRS_STACK");
  }
  return $success;
}

=head2 configure_ASM

   Creates or updates ASM

=head3 Parameters

   None

=head3 Returns

  TRUE  - ASM configuration was     created or updated
  FALSE - ASM configuration was not created or updated

=head3 Notes

  This will start ASM as part of the configuration if it is successful

=cut

sub configure_ASM {
   my $status;
   my $ASMDISKS             = $CFG->params('ASM_DISKS');
   my $ASM_DISCOVERY_STRING = $CFG->params('ASM_DISCOVERY_STRING');
   my $success              = TRUE;
   my $diskgroup;

   trace ("Configuring ASM via ASMCA");

   # Do not change the order of these parameters as asmca requires the
   # parameters to be in a specific order or it will fail
   my @runasmca = (catfile ($CFG->ORA_CRS_HOME, "bin", "asmca"), '-silent');
   if ($CFG->params('ASM_DISK_GROUP') ){
      $diskgroup = $CFG->params('ASM_DISK_GROUP');
      if ($diskgroup =~ /\$/) {
         # if diskgroup contains '$', put single-quotes around it
         quoteDiskGroup($diskgroup);
         push @runasmca, '-diskGroupName', "'$diskgroup'";
      }
      else {
         push @runasmca, '-diskGroupName', $diskgroup;
      }
   }

   # When this is run as superuser
   if ($CFG->params('ASM_DISKS')) {
      push @runasmca, '-diskList', "'$ASMDISKS'";
   }

   if ($CFG->params('ASM_REDUNDANCY')) {
      push @runasmca, '-redundancy', $CFG->params('ASM_REDUNDANCY');
   }

   if ($CFG->params('ASM_DISCOVERY_STRING')) {
      push @runasmca, '-diskString', "'$ASM_DISCOVERY_STRING'";
   }

   if (isFirstNodeToStart()) {
      push (@runasmca, ('-configureLocalASM'));

      # $diskgroup is single-quoted above
      push @runasmca, '-passwordFileLocation', "+$diskgroup/orapwASM"; 
   }

   if (isReusedg()) {
      push (@runasmca, ('-reuseDiskGROUP'));
   }

   if ($CFG->params('ASM_AU_SIZE')) {
      push @runasmca, '-au_size', $CFG->params('ASM_AU_SIZE');
   }

   if ($CFG->defined_param('ORATAB_LOC')) {
      push (@runasmca, ('-oratabLocation'), $CFG->params('ORATAB_LOC'));
   }

   trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @runasmca");
   $status       = run_as_user($CFG->params('ORACLE_OWNER'), @runasmca);
   my $asmca_log = catdir($CFG->params('ORACLE_BASE'), 'cfgtoollogs', 'asmca');

   if ($status != 0) {
      $success = FALSE;
      print_error(184);
      trace("See asmca logs at $asmca_log for details.");
   }
   else {
      #Bug - 10417856
      if (check_service("ora.asm", 20)) {
         $success = TRUE;
      }
      else {
         print_error(12);
         $success = FALSE;
      }
   }

   return $success;
}

=head2 configure_ASM_oda

   Creates or updates ASM for ODA

=head3 Parameters

   None

=head3 Returns

  TRUE  - ASM configuration was     created or updated
  FALSE - ASM configuration was not created or updated

=head3 Notes

  This will start ASM as part of the configuration if it is successful

=cut
sub configure_ASM_oda
{
  my $ORA_CRS_HOME = $CFG->ORA_CRS_HOME;
  my $success = TRUE;
  my $status;
  my $ASMDISKS = $CFG->params('ASM_DISKS');
  my $ASM_DISCOVERY_STRING = $CFG->params('ASM_DISCOVERY_STRING');

  trace ("Configuring ASM for ODA");

  my @runodaDg = ("/opt/oracle/oak/onecmd/runodaDg.pl");
  if ($CFG->params('ASM_DISK_GROUP') ){
     my $diskgroup = $CFG->params('ASM_DISK_GROUP');
     if ($diskgroup =~ /\$/) {
        # if diskgroup contains '$', put single-quotes around it
        quoteDiskGroup($diskgroup);
        push @runodaDg, '-d', "'$diskgroup'";
     }
     else {
        push @runodaDg, '-d', $diskgroup;
     }
  }

  # When this is run as superuser
  if ($CFG->params('ASM_DISKS')) {
    push @runodaDg, '-l', "'$ASMDISKS'";
  }

  if ($CFG->params('ASM_REDUNDANCY')) {
    push @runodaDg, '-r', $CFG->params('ASM_REDUNDANCY');
  }

  if ($CFG->params('ASM_DISCOVERY_STRING')) {
    push @runodaDg, '-s', "'$ASM_DISCOVERY_STRING'";
  }
  
   push @runodaDg, '-o', $ORA_CRS_HOME;


  trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @runodaDg");
  $status       = run_as_user($CFG->params('ORACLE_OWNER'), @runodaDg);

  if ($status != 0) {
     $success = FALSE;
     error ("Configuration of ODA ASM ... failed");
  }
  else
  {
    #Bug - 10417856
    if (check_service("ora.asm", 20)) {
      $success = TRUE;
    }
    else {
      error ("The ora.asm resource is not ONLINE");     
      $success = FALSE;
    }
  }

  return $success;
}

sub add_voting_disks
{
      my $ASM_DISK_GROUP = $CFG->params('ASM_DISK_GROUP');
      my $success;
      # Depending on whether using ASM or not, create
      # accordingly
      if ($CFG->ASM_STORAGE_USED)
      {
        $success = CSS_add_vfs($CFG, "+$ASM_DISK_GROUP");
      } else {
        $success = CSS_add_vfs($CFG,
                               split(',', $CFG->params('VOTING_DISKS')));
      }
      return $success;
}

sub perform_configNode
{
  my $ckptStatus;
  my $ckptName = "ROOTCRS_NODECONFIG";

  if (isCkptexist($ckptName))
  {
    $ckptStatus = getCkptStatus($ckptName);
    trace("'$ckptName' state is $ckptStatus");

    if ($ckptStatus eq CKPTSUC) {
       trace("CRS node configuration resources are already configured");
       $CFG->wipCkptName("ROOTCRS_STACK");
       return SUCCESS;
    }
  }

  writeCkpt($ckptName, CKPTSTART);
  $CFG->wipCkptName($ckptName);

  configNode();
}

sub configNode
#-------------------------------------------------------------------------------
# Function: Configure node
# Args    : none
# Returns : TRUE  if success
#           FALSE if failed
#-------------------------------------------------------------------------------
{
   trace ("Configuring node");
   my $DHCP_flag     = FALSE;
   my $success       = TRUE;
   my $ckptName      = "ROOTCRS_NODECONFIG";
   my $asmcainitflag = FALSE;

   # set DHCP_flag to TRUE if it's DHCP
   my $crs_nodevips     = $CFG->params('CRS_NODEVIPS');
   $crs_nodevips        =~ s/'//g; # ' in comment to avoid confusion of editors.
   $crs_nodevips        =~ s/"//g; # remove " on Windows
   $crs_nodevips        =~ s/\/\*/\//g; # handle different interface case
   my @crs_nodevip_list = split (/\s*,\s*/, $crs_nodevips);

   if ($crs_nodevip_list[0] =~ /\bAUTO/) {
      $DHCP_flag = TRUE;
   }

   # configure new node
   if (isAddNode($CFG->HOST, $CFG->params('NODE_NAME_LIST'))) {
      trace("Performing new node configurations");
      $success = configNewNode($CFG->HOST);
      if ($success) {
         writeCkptProperty($ckptName, $CFG->HOST, "ADDED_NODE");
         writeCkpt($ckptName, CKPTSUC);
      }
      else {
         writeCkpt($ckptName, CKPTFAIL);
         die(dieformat(277));
      }

      return $success;
   }

   # configure fresh install node
   my $upgrade_option;
   my @nodevip;
   my @nodes_to_add;
   my @nodes_to_start;

   # add nodeapps
   my @node_list = split (',', $CFG->params('NODE_NAME_LIST'));
   my $ix        = 0;

   foreach my $node (@node_list) {
      if ($CFG->HOST =~ /$node$/i) {
         push @nodevip, $crs_nodevip_list[$ix];
         push @nodes_to_add, $node;
         last; # done for this node
      } else {
         $ix++;
      }
   }

   if (isFirstNodeToStart()) {
      upgradeModelResType();

      $success = add_Nodeapps($upgrade_option, \@nodevip, $DHCP_flag,
                              \@nodes_to_add, \@nodes_to_start);
      if ($success != TRUE) {
         writeCkpt($ckptName, CKPTFAIL);
         die(dieformat(286));
      }

      $success = configFirstNode($DHCP_flag, \@nodes_to_start);
      if ($success != SUCCESS) {
         writeCkpt($ckptName, CKPTFAIL);
         die(dieformat(287));
      }
   }
   else {
      if ($DHCP_flag) {
         push @nodes_to_start, $nodes_to_add[0];
      }
      else {
         $success = add_Nodeapps($upgrade_option, \@nodevip, $DHCP_flag,
                                 \@nodes_to_add, \@nodes_to_start);
         if ($success != SUCCESS) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(286));
         }
      }

      if ($success) {
         $success = start_Nodeapps($DHCP_flag, \@nodes_to_start);
      }
         if ($success != SUCCESS) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(288));
         }

      # wait for cluster to come up
      check_service("cluster", 60);
   }

   writeCkpt($ckptName, CKPTSUC);
   $CFG->wipCkptName("ROOTCRS_STACK");
   return $success;
}

sub configNewNode
#---------------------------------------------------------------------
# Function: Configure nodeapps for new node
# Args    : [0] new node
#           [1] DHCP tag to indicate if DHCP is used
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   my $newnode      = $_[0];
   my $srvctl       = crs_exec_path('srvctl');
   my $run_as_owner = FALSE;
   my (@capout, @cmd, $status);
   trace ("Configure Nodeapps for new node=$newnode");

   if (isBigCluster() && isRimNode())
   {
     trace("Skip configuring nodeapps on rim node");
     return TRUE;
   }

   if ($CFG->params('CRS_DHCP_ENABLED') eq 'false') {
      # get VIP
      my $hostvip = getHostVIP($newnode);
      if (! $hostvip) {
         die(dieformat(276));
      }

      # add vip
      $status = srvctl($run_as_owner,
                       "add vip -n $newnode -k 1 -A \"$hostvip\" ");

      if (${status}) {
         trace ("add vip on node=$newnode ... success");
      }
      else {
         print_error(180, "add vip -n $newnode -k 1 -A \"$hostvip\"", $status);
         error ("add vip on node=$newnode ... failed");
         return FALSE;
      }
   }

   # wait for vip resource to exist before start vip
   if ($CFG->params('CRS_DHCP_ENABLED') eq 'true') {
      my $vip_exists = waitForVipRes();
      if (! $vip_exists) {
         print_error(10);
         return FALSE;
      }
   }

   # start vip
   $status = srvctl($run_as_owner, "start vip -i $newnode");

   if (${status}) {
      trace ("start vip on node:$newnode ... success");
   }
   else {
      print_error(108, $newnode);
      return FALSE;
   }

   return TRUE;
}

sub getHostVIP
#---------------------------------------------------------------------
# Function: Get Host's VIP from CLUSTER_NEW_VIPS
# Args    : [0] Hostname
# Returns : Host's VIP
#---------------------------------------------------------------------
{
   my $hostname     = $_[0];
   my @new_hosts    = split (/,/, $CFG->params('CLUSTER_NEW_HOST_NAMES'));
   my @new_vips     = split (/,/, $CFG->params('CLUSTER_NEW_VIPS'));
   my $nbr_of_hosts = scalar(@new_hosts);
   my $nbr_of_vips  = scalar(@new_vips);
   my $srvctl       = crs_exec_path('srvctl');
   my $ix           = 0;
   my ($hostVIP, $netmask, $if, $prefixlen6);

   if (($CFG->params('CRS_DHCP_ENABLED') ne 'true') &&
       ($nbr_of_hosts != $nbr_of_vips))
   {
      die(dieformat(274));
   }

   # check version number
   # use '-S 1' if version is greater than 11.1. Otherwise use '-a'
   my @version = get_crs_version($CFG->ORA_CRS_HOME);

   if (1 == versionComparison(join(".", @version), "11.1.0.0.0"))
   {
      # use '-S 1'
      open OPUT, "$srvctl config nodeapps -S 1|";
      my @viplist = grep(/netmask=/, <OPUT>);
      close OPUT;

      trace("viplist = '@viplist'");
      if (scalar(@viplist) > 0) {
         trace ("viplist='$viplist[0]'");

         # get new VIP
         $ix = 0;
         foreach my $host (@new_hosts) {
            chomp $host;
            if ($hostname =~ /$host$/i) {
               last;
            }
            $ix++;
         }

         # append netmask/if to new vip
         # For IPV4, netmask is pouplated and prefixlen6 is empty.
         # For IPV6, netmask is emtpy but we have prefixlen6 pouplated.
         my @list = split(/} +/, $viplist[0]);
         foreach (@list) {
            trace("IP list=$_");
            if ($_ =~ /netmask=/) {
               $netmask = parseText ($_);
            }
            elsif ($_ =~ /interfaces=/) {
               $if = parseText ($_);
            }
            elsif ($_ =~ /prefixlen6=/)
            {
               $prefixlen6 = parseText($_);
            }
         }

         trace("netmask = $netmask");
         trace("interface = $if");
         trace("prefixlen6 = $prefixlen6");
      }
   }
   else {
      # use '-a'
      open OUTPUT, "$srvctl config nodeapps -a|";
      my @viplist = (<OUTPUT>);
      chomp @viplist;
      close OUTPUT;
      trace ("VIPList=$viplist[1]");

      if ($viplist[1] =~ /:/) {
         # get new VIP
         my @VIPs = split (/\//, $viplist[1]);

         if (scalar(@VIPs) >= 5) {
            $if      = $VIPs[5];
            trace("interface = $if");
         }

         if (scalar(@VIPs) >= 4) {
            $netmask = $VIPs[4];
            trace("netmask = $netmask");
         }

         $ix = 0;
         foreach my $host (@new_hosts) {
            chomp $host;
            if ($hostname =~ /$host$/i) {
               last;
            }

            $ix++;
         }
      }
   }

   # append netmask/if to new vip
   if ((! $netmask) && $prefixlen6)
   {
     $netmask = $prefixlen6;
   }

   if ($netmask) {
      $hostVIP = $new_vips[$ix] . "/" . $netmask;
      if ($if) {
         # translate / replace ":" with "|" in the interface string
         $if =~ tr/[:]/[|]/;
         $hostVIP = $hostVIP . "/" . $if;
      }
   }

   sub parseText {
      # extract netmask and interface
      my $line = $_[0];
      $line =~ s/{//g;
      $line =~ s/}//g;
      my ($dummy, $text) = split (/=/, $line);
      chomp $text;

      return $text;
   }

   trace("hostVIP = $hostVIP");

   return $hostVIP;
}

sub start_asm
{
   my $run_as_oracle_owner = TRUE;
   my $status = srvctl($run_as_oracle_owner, "start asm");

   if ($status) {
      trace ("start asm ... success");
   } 
   else {
      print_error(113);
      return FALSE;
   }

   return TRUE;
}

sub stop_asm
{
   my $run_as_oracle_owner = TRUE;
   my $status = srvctl($run_as_oracle_owner, "stop asm -f");

   if ($status) {
      trace ("stop asm ... success");
   }
   else {
      print_error(114);
      return FALSE;
   }

   return TRUE;
}

sub backupOLR
#-------------------------------------------------------------------------------
# Function: Backup OLR 
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
   my $ocrconfig = catfile($CFG->params('ORACLE_HOME'), 'bin', 'ocrconfig');
   my $cmd = "$ocrconfig -local -manualbackup";
   my @out = system_cmd_capture($cmd);
   my $rc = shift @out;

   if ($rc == 0) {
      trace ("Successfully generated OLR backup");
   } else {
      trace ("Failed to generate OLR backup");
      return FAILED;
   }

   return SUCCESS;
}

sub backupOCR
#------------------------------------------------------------------------------- 
## Function: Backup OCR 
## Args    : none
## Returns : SUCCESS or FAILED
##------------------------------------------------------------------------------- 
{
   my $ocrconfig = catfile($CFG->params('ORACLE_HOME'), 'bin', 'ocrconfig');
   my $cmd = "$ocrconfig -manualbackup";
   my @out = system_cmd_capture($cmd);
   my $rc = shift @out;

   if ($rc == 0) {
      trace ("Successfully generated OCR backup");
   } else {
      trace ("Failed to generate OCR backup");
      return FAILED;
   }

   return SUCCESS;
}

sub configureCvuRpm
#------------------------------------------------------------------------------
# Function:  Install cvuqdisk rpm on Linux 
# Args    :  none
#-------------------------------------------------------------------------------
{
   if ($CFG->platform_family eq "unix") {
      s_configureCvuRpm();
   }
}

sub configureRemoteNodes
#-------------------------------------------------------------------------------
# Function: Configure all remote nodes for Windows
# Args    : none
#-------------------------------------------------------------------------------
{
   if (($CFG->platform_family eq "windows") &&
       (! $CFG->REMOTENODE) && (! $CFG->addnode))
   {
      if (!isCkptexist("ROOTCRS_REMOTENODES")) {
         trace("Writing checkpoint for remotenode actions");
         writeCkpt("ROOTCRS_REMOTENODES", CKPTSTART);
         $CFG->wipCkptName("ROOTCRS_REMOTENODES");
      }

      if (!isCkptSuccess("ROOTCRS_REMOTENODES")) {
         $CFG->wipCkptName("ROOTCRS_REMOTENODES");
         writeCkpt("ROOTCRS_REMOTENODES", CKPTSTART);

         # for windows only, configure all remote nodes
         if (s_configureAllRemoteNodes()) {
            writeCkpt("ROOTCRS_REMOTENODES", CKPTSUC);
            trace ("Configure Oracle Grid Infrastructure on remote nodes... succeeded");
         }
         else {
            writeCkpt("ROOTCRS_REMOTENODES", CKPTFAIL);
            trace ("Configure Oracle Grid Infrastructure on remote nodes... failed");
            exit 100;
         }
      }
   }

   return SUCCESS;
}

sub precheck_siha
{
   if (isPrereqIgnored()) {
      print_error(363);
   }

   # validate RAC_ON/RAC_OFF
   if (! is_dev_env () && (! isRAC_appropriate())) {
      exit 1;
   }
}

sub isCRSAlreadyConfigured
#-------------------------------------------------------------------------------
# Function: Check if CRS is already configured on this node
# Args    : none
# Return  : TRUE  if CRS is     already configured
#           FALSE if CRS is not already configured
#-------------------------------------------------------------------------------
{
   my $crshome;
   my $olr_exists      = FALSE;
   my $localOCR_exists = FALSE;
   my $crs_exists      = s_check_CRSConfig($CFG->HOST,
                                           $CFG->params('ORACLE_OWNER'));

   if ($CFG->platform_family eq "windows") {
      $crshome = s_get_olr_file ("crs_home");
      if ($crshome) {
         $olr_exists = TRUE;
      }
   }
   else {
      if (-e $CFG->params('OLRCONFIG')) {
         $crshome = s_get_olr_file ("crs_home");
         if ($crshome) {
            $olr_exists = TRUE;
         }
      }
   }

   if ($CFG->platform_family eq "windows") {
      $localOCR_exists = local_only_config_exists();
   }
   else {
      if (-e $CFG->params('OCRCONFIG')) {
         $localOCR_exists = local_only_config_exists();
      }
   }

   if ($olr_exists && $crs_exists) {
      print_error(350);
      print_error(352, $crshome);
      trace ("Please deconfigure before proceeding with the " .
             "configuration of new home. ");
      return TRUE;
   }
   elsif ((! $olr_exists) && (! $crs_exists)) {
      trace ("CRS is not yet configured. Hence, will proceed to configure CRS");
      return FALSE;
   }
   elsif ($CFG->UPGRADE && (! $olr_exists)) {
      return FALSE;
   }
   elsif (!$CFG->UPGRADE && (! $olr_exists) && ($localOCR_exists)) {
      return FALSE;
   }
   else {
      my $rootscript = "root.sh";
      my $rootdeconfig = "rootcrs.pl";
      if ($CFG->platform_family eq "windows") {
         $rootscript = "gridconfig.bat";
      }

      if ($CFG->SIHA) {
         $rootdeconfig = "roothas.pl";
      }

      print_error(351);
      trace ("Deconfigure the existing cluster configuration before starting");
      trace ("to configure a new Clusterware");
      print_error(353, $CFG->params('ORACLE_HOME'), $rootdeconfig, $rootscript);
      return TRUE;
   }
}

sub prepare_to_start_siha
{
   my $local_config_exists = local_only_config_exists();

   # delete olr file if exists
   if (-e $CFG->OLR_LOCATION) {
      if ($CFG->DEBUG) { trace ("unlink " . $CFG->OLR_LOCATION); }
      unlink ($CFG->OLR_LOCATION) || print_error(102, $CFG->OLR_LOCATION, $!);
   }

   validate_olrconfig($CFG->OLR_LOCATION,
                      $CFG->ORA_CRS_HOME) || die(dieformat(292));

   if (! $local_config_exists) {
      createLocalOnlyOCR();
   }

   upgrade_local_OCR();

   create_OLR_keys();
   
   start_or_upgrade_siha($local_config_exists);
}

sub start_or_upgrade_siha
{
   my $local_config_exists = $_[0];
   my $del_ohasd_res = $_[1];
   my $status;

   # check if older version SI CSS exists, by looking in OCRLOC
   $CFG->restart_css(FALSE);

   if ($local_config_exists) {
      # If the stack is up, keep track of this and stop it now
      if ($CFG->restart_css(local_only_stack_active())) {
         if (!stop_local_only_stack()) {
            die(dieformat(319));
         }

         #Skipping the steps below on Windows.
	 if ($CFG->platform_family ne "windows") {
	   # Bug 8255312 - In single instance asm upgrade to siha
	   # these directories gets removed when stopping the old css
	   # by localconfig -delete
	   my $SCRDIR = catfile ($CFG->params('SCRBASE'), $CFG->HOST);
	   my @NS_DIRS  = ("/var/tmp/.oracle", "/tmp/.oracle");
	   if (! -d $CFG->params('SCRBASE')) {
		trace("creating dir $SCRDIR");
		create_dir($SCRDIR);
		s_set_ownergroup($CFG->SUPERUSER,
		                 $CFG->params('ORA_DBA_GROUP'), $SCRDIR);
           }

	   for my $NSDIR (@NS_DIRS)
	   {
	      if (! -d $NSDIR) {
	      trace("creating dir $NSDIR");
	      create_dir($NSDIR);
	      s_set_ownergroup($CFG->SUPERUSER,
	                        $CFG->params('ORA_DBA_GROUP'), $NSDIR);
              s_set_perms ("01777", $NSDIR);
	      }
	   }
	 }
      }

      # Clean up local endpts used by CSSD, bug 891745
      CSS_Clean_Local_Endpts($CFG);

      if (!migrate_dbhome_to_SIHA()) {
         die(dieformat(320));
      }
   }
   # Confgiure local only OCR and pin the node
   else {
      if (! configure_OCR()) {
         print_error(156);
         trace ("Creating local-only OCR ... failed");
         exit;
      }
      else {
         trace ("Creating local-only OCR ... succeeded");
      }
   }

   # Initialize the SCR settings.
   s_init_scr ();

   # register OHASD service/daemon with OS
   trace ("Registering ohasd");
   register_ohasd() || die(dieformat(317));

   # Need to start OHASD as non-$SUPERUSER, i.e., as crsuser, which in this case
   # would be ORACLE_OWNER
   trace ("Starting ohasd");
   start_siha_ohasd();

   # Check if the service/daemon has started
   trace ("Checking ohasd");
   my $ohasd_running = check_service ("ohasd", 360);

   if ($ohasd_running) {
      trace ("ohasd started successfully");
   }
   else {
      print_error(199);
      exit 1;
   }

   trace ("Creating HA resources and dependencies");

   my $owner = $CFG->params('ORACLE_OWNER');
   if ((! is_dev_env()) && ($CFG->platform_family eq "windows")) {
      $owner = '';
   }

   #Delete ohasd resources for SIHA for 11.2 upgrade
   if ($del_ohasd_res) {
      my $crsctl = crs_exec_path('crsctl');
      # There should not be crs and ctss in SIHA,
      # so remove both of them from 11.2
      system_cmd_capture($crsctl, "delete", "type", "ora.crs.type", "-init");
      system_cmd_capture($crsctl, "delete", "type", "ora.ctss.type", "-init");
      $status = configure_hasd('has', $CFG->HOST, $owner, $owner,
                               $CFG->params('ORA_DBA_GROUP'), "delete");
   }

   $status = configure_hasd('has', $CFG->HOST, $owner, $owner,
                            $CFG->params('ORA_DBA_GROUP'));


   if ($status) {
      trace ("Successfully created HA resources for HAS daemon and ASM");
   }
   else {
      print_error(200);
      exit 1;
   }
}

sub start_siha_ohasd
{
  my $status = start_service("ohasd", $CFG->HAS_USER);
 
  if (! $status)
  {
    if (is_dev_env() && $CFG->UPGRADE)
    {
      trace("This is a dev env for SIHA upgrade");
      my $av;
      my $retry = 0;
      my $OCRDUMPBIN = catfile($CFG->ORA_CRS_HOME, 'bin', 'ocrdump');
      my @output;

      while ($retry++ < 60)
      {
        open (OCRDUMP, "$OCRDUMPBIN -local -noheader -stdout -keyname ".
          "SYSTEM.version.activeversion |");
        @output = <OCRDUMP>;
        close(OCRDUMP);

        trace("ocrdump output for active version is @output"); 
        my @txt = grep(/ORATEXT/, @output);
        if (scalar(@txt) > 0)
        {
          my ($key, $ver) = split(/:/, $txt[0]);
          $av = trim($ver);
          trace("The active version is $av");
        }

        if (-1 == versionComparison($av, "12.1.0.0.0"))
        {
          trace("Waiting for AV to become new");
          sleep(10);
        }
        else { last; }
      }
 
      my $success;
      if (-1 != versionComparison($av, "12.1.0.0.0"))
      {
        trace("Resource and type conversion completed. Restart ohasd");
        my $CRSCTL  = catfile($CFG->ORA_CRS_HOME, "bin", "crsctl");
        @output = system_cmd_capture($CRSCTL, 'start', 'has');
        $status = shift @output;
        if ((0 == $status) || (scalar(grep/4640/, @output) > 0))
        {
          $success = TRUE;
          trace("Oracle High Availability Services is up now");
        }
      }

      if (! $success) { die(dieformat(318)); }
    }
    else
    {
     die(dieformat(318));
    }
  }

  trace("Successfully started ohasd for SIHA");
}

sub local_only_config_exists
{
  my $found      = FALSE;
  my $local_only = s_get_config_key("ocr", "local_only");
  my $ocrcfg_loc = s_get_config_key("ocr", "ocrconfig_loc");
  my $db_home;

  if ($local_only =~ m/true/i) {
     $CFG->oldconfig('OCRCONFIG', $ocrcfg_loc);

     # get older version DBHOME path
    if ($ocrcfg_loc =~ m/(.+).cdata.localhost.local.ocr/) {
       $db_home = $1;
       if ($db_home) {
          $CFG->oldconfig('DB_HOME', $db_home);
          $found = TRUE;
          trace ("local_only config exists");
          #get old  releaseversion of SIHA/SI-CSS.
          my @oldCrsVer = get_has_version($db_home);
          $CFG->oldconfig('ORA_CRS_VERSION', \@oldCrsVer);
       }
       else {
          print_error(181);
       }
    }
    else {
       print_error(7);
    }
  }

  return $found;
}

sub get_has_version
{
   my $home = $_[0];
   my @ver  = (10, 1, 0, 0, 0);
   my $crsctl;

   if (! defined $home) {
      $crsctl = crs_exec_path('crsctl');
   } else {
      $crsctl = catfile($home, 'bin', 'crsctl' );
   }

   # run "crsctl query crs releaseversion" -- local CSS must be up
   # Example output:
   # Oracle Clusterware  version on the local node is [11.2.0.0.2]
   my @cmd = ($crsctl, 'query', 'has', 'releaseversion');
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;
   my $verstring;

   # if succeeded, parse to ver numbers, output must be a single line,
   # version is 5 numbers, major to minor (see above)
   if ($rc == 0) {
      $verstring = getVerInfo($out[0]);
      @ver          = split(/\./, $verstring);
      trace( "Got High Availability release version: ".join('.', @ver) );
   }
   elsif ($rc != 0) {
      @cmd = ($crsctl, 'query', 'crs', 'releaseversion');
      @out = system_cmd_capture(@cmd);
      $rc  = shift @out;
      if ($rc == 0) {
         $verstring = getVerInfo($out[0]);
         @ver          = split(/\./, $verstring);
         trace( "Got Single Instance CSS release version: ".join('.', @ver) );
      }
      elsif ($rc != 0) {
         @cmd = ($crsctl, 'query', 'crs', 'activeversion');
         @out = system_cmd_capture(@cmd);
         $rc  = shift @out;
        
         if ($rc == 0) {
           $verstring = getVerInfo($out[0]);
           @ver          = split(/\./, $verstring);
           trace( "Got Single Instance CSS release version: ".join('.', @ver) );
         }
         else {
           my $mycmd = join(' ', @cmd);
           print_error(180, $mycmd, $rc);
           trace ("@cmd ... failed rc=$rc with message:\n @out \n");  
         }
      }
   }
   else {
      my $mycmd = join(' ', @cmd);
      print_error(180, $mycmd, $rc);
      trace ("@cmd ... failed rc=$rc with message:\n @out \n");
   }

   return @ver;
}

sub createLocalOnlyOCR
#-------------------------------------------------------------------------------
# Function:  Create local-only OCR
# Args    :  none
#-------------------------------------------------------------------------------
{
   trace ("create Local Only OCR...");
   my $owner      = $CFG->params('ORACLE_OWNER');
   my $dba_group  = $CFG->params('ORA_DBA_GROUP');
   my $local_ocr  = catfile ($CFG->params('ORACLE_HOME'), "cdata",
                             "localhost", "local.ocr");
   s_createLocalOnlyOCR();

   # create local.ocr and set ownergroup
   open (FILEHDL, ">$local_ocr") or die(dieformat(207, $local_ocr, $!));
   close (FILEHDL);
   s_set_ownergroup ($owner, $dba_group, $local_ocr)
                or die(dieformat(152, $local_ocr));
   s_set_perms ("0640", $local_ocr)
                or die(dieformat(153, $local_ocr));


   # validate local.ocr and update ocr.loc
   validate_ocrconfig ($local_ocr, $CFG->SIHA)
                or die(dieformat(301));

}

sub local_only_stack_active
{
  $ENV{'ORACLE_HOME'} = $CFG->oldconfig('DB_HOME');

  my $restart_css = FALSE;
  # check if older version SI CSS is running
  my $OLD_CRSCTL = catfile ($CFG->oldconfig('DB_HOME'), "bin", "crsctl");
  my $status = system ("$OLD_CRSCTL check css");

  if (0 == $status) {
    # set flag to restart SIHA CSS before we're done
    $restart_css = TRUE;
  }

  $ENV{'ORACLE_HOME'} = $CFG->params('ORACLE_HOME');
  return $restart_css;
}

sub stop_local_only_stack
{
   my $OLD_SIHOME    = $CFG->oldconfig('DB_HOME');
   my $stack_stopped = SUCCESS;
   my $status;

   #Bug 8280425. Take a backup of ocr.loc and local.ocr
   #before invoking localconfig -delete
   my $local_ocr      = catfile($OLD_SIHOME, 'cdata', 'localhost', 'local.ocr');
   my $local_ocr_save = catfile($OLD_SIHOME, 'cdata', 'localhost', 'local.ocr.save');
   my ($ocr_loc, $ocr_loc_save);

   if ($CFG->platform_family eq "windows") {
      $ocr_loc      = $CFG->params('OCRLOC');
      $ocr_loc_save = $CFG->params('OCRLOC') . '.save';
      trace("backing up $ocr_loc registry");
      s_copyRegKey($ocr_loc, $ocr_loc_save);
   }
   else {
      $ocr_loc      = catfile ($CFG->params('OCRCONFIGDIR'), 'ocr.loc');
      $ocr_loc_save = catfile ($CFG->params('OCRCONFIGDIR'), 'ocr.loc.save');
      trace("backing up $ocr_loc");
      if (copy_file ($ocr_loc, $ocr_loc_save) != SUCCESS) {
        print_error(165, $ocr_loc);
      }
   }

   # backing up local_ocr
   trace("backing up $local_ocr");
   if (copy_file ($local_ocr, $local_ocr_save) != SUCCESS) {
      print_error(165, $local_ocr);
   }

   if ($CFG->platform_family eq "windows") {
      s_stopDeltOldASM();
      s_stopService("OracleCSService");
      if (s_isServiceRunning("OracleCSService")) {
         s_stopService("OracleCSService");
      }

      s_deltService("OracleCSService")
   }

  # stop old SI CSS
   trace ("Stopping older version SI CSS");
   my $OLD_LOCALCONFIGBIN = catfile ($CFG->oldconfig('DB_HOME'),
                                     "bin", "localconfig");
   $status = system ("$OLD_LOCALCONFIGBIN delete");
   if ($status == 0) {
      trace ("Older version SI CSS successfully stopped/deconfigured");
   }
   else {
      $stack_stopped = FAILED;
      print_error(166);
   }

   # Bug 8280425 'localconfig -delete' removes the ocr.loc and local.ocr
   # restore the same.
   if ($CFG->platform_family eq "windows") {
      trace("restoring $ocr_loc registry");
      s_copyRegKey($ocr_loc_save, $ocr_loc);
   }
   else {
      trace("restoring $ocr_loc");
      if (copy_file ($ocr_loc_save, $ocr_loc) != SUCCESS) {
         print_error(167, $ocr_loc);
      }
   }

  # restoring local_ocr
   trace("restoring $local_ocr");
   if (copy_file ($local_ocr_save, $local_ocr ) != SUCCESS) {
      print_error(167, $local_ocr);
   }

   if ($CFG->platform_family eq "unix") {
      s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                       $CFG->params('ORA_DBA_GROUP'),
                       $ocr_loc) || die(dieformat(152, $ocr_loc));
      s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                       $CFG->params('ORA_DBA_GROUP'), 
                       $local_ocr) || die(dieformat(152, $local_ocr));
   }

   return $stack_stopped;
}

####---------------------------------------------------------
#### Function name : migrate_dbhome_to_SIHA
# ARGS 0:
# This routine does the operations in the following sequence.
# 1) Take a backup copy of older ocr file.
# 2) Update the location of ocr.loc
# 3) touch and change file permissions.
# 4) Create necessary configuration with 'crsctl pin css' command.

sub migrate_dbhome_to_SIHA {
  my ($db_home, $status);
  my $OCRCONFIGBIN = crs_exec_path("ocrconfig");
  my $CRSCTLBIN    = crs_exec_path("crsctl");
  my $ocrcfg_loc   = $CFG->oldconfig('OCRCONFIG');
  my $copy_lococr  = catfile($CFG->params('ORACLE_HOME'),
                             'cdata', 'localhost', 'localsiasm.ocr');
  my $new_lococr   = catfile($CFG->params('ORACLE_HOME'),
                             'cdata', 'localhost', 'local.ocr');
  my $ret          = FAILED;
  $ENV{'NLS_LANG'} = $CFG->params('LANGUAGE_ID');

  # copy over older local-only OCR to SIHA home
  if (defined $ocrcfg_loc) {
     if (copy_file ($ocrcfg_loc, $copy_lococr) != SUCCESS) {
        print_error(106);
     }
  }

  # Now touch, set owner and perm
  if (!(-e $new_lococr)) {
    if ($CFG->DEBUG) { trace ("Creating empty file $new_lococr");}
    # create an empty file
    open (FILEHDL, ">$new_lococr") or die(dieformat(255, $new_lococr, $!));
    close (FILEHDL);
  }
  # Set ownership/group
  s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                    $CFG->params('ORA_DBA_GROUP'),
                    $new_lococr) || die(dieformat(152, $new_lococr));

  # Set permissions, if specified
  s_set_perms ("0640", $new_lococr) || die(dieformat(153, $new_lococr));

  # update ocr.loc
  if (defined $ocrcfg_loc) {
     if (0 !=  system_cmd($OCRCONFIGBIN, '-repair',
                                         '-replace', $ocrcfg_loc,
                                         '-replacement', $new_lococr)) {
        print_error(155);
     }
  }

  # Create configuration needed for compatibility with older DBs (10.x, 11.1).
  $ret = configure_OCR();
  if (!$ret)
  {
    print_error(156);
    trace ("Creating local-only OCR ... failed");
  }
  else
  {
    trace ("Creating local-only OCR ... succeeded");
  }

  return $ret;
}

################################################################################
# Function: Install XAG component
################################################################################
sub install_xag
{
  if (is_dev_env())
  {
    return SUCCESS;
  }

  if ($CFG->platform_family ne "unix")
  {
    trace("XAG is not yet supported on " . $CFG->platform_family);
    return SUCCESS;
  }

  trace("Installing XAG component");
  my $XAG_HOME = catdir($CFG->ORA_CRS_HOME, 'xag');
  my $CRS_HOME = $CFG->ORA_CRS_HOME;
  trace("xag_home=$XAG_HOME crs_home=$CRS_HOME");

  # set up substitution vars table
  my %subsVars = ();
  $subsVars{XAG_HOME} = $XAG_HOME;
  $subsVars{CRS_HOME} = $CRS_HOME;
  $subsVars{PERL_HOME} = catdir($CRS_HOME, 'perl');
  $subsVars{DEPLOYMENT} = 'bundled';

  if ($CFG->platform_family eq "unix")
  {
    $subsVars{XAG_OLR_LOC} = $CFG->params('OLRCONFIG');
  }

  # substitute vars in sbs and template directories
  trace("substitute vars in sbs and template directories");
  my $sbsDir = catdir($XAG_HOME, 'sbs');
  my @sbsFiles = glob(catfile($sbsDir, "*"));
  my $templateDir = catdir($XAG_HOME, 'template');
  push @sbsFiles, glob(catfile($templateDir, "*"));
  my $binDir = catdir($CRS_HOME, 'bin');

  # instantiate variables
  foreach my $sbsFile (@sbsFiles)
  {
    my @sbsContent = read_file($sbsFile);
    my $dstfile = basename($sbsFile);
    $dstfile =~ s/\.sbs//g; # strip off .sbs suffix
    my $dstpath = catfile ($binDir, $dstfile);
    trace("sbsFile=$sbsFile dstPath=$dstpath");

    open (DSTPATH, ">${dstpath}") || die("Can't open file", $dstpath, $!);

    foreach my $line (@sbsContent)
    {
      if ($line !~ /^#|^\s*$/)
      {
        # If it contains a pattern of the form '%foo%' AND a mapping exists
        # for 'foo', replace '%foo%' with the corresponding value.
        my $rexp = "[a-zA-Z_]+";
        my @matchlist = $line =~ /%(${rexp})%/g;
        my $sub;
        foreach my $match (@matchlist)
        {
          if (exists $subsVars{$match})
          {
            $sub = $subsVars{$match};
            $line =~ s/%(${match})%/$sub/g;
          }
          else
          {
            trace("Can't substitue variable=$match file=$sbsFile");
          }
        }
      }

      print DSTPATH "$line";
    }

    close (DSTPATH);
    s_set_perms(755, $dstpath);
  }

  # move template type files from bin to template directory
  my @binFiles = glob(catfile($binDir, "*.type"));
  foreach my $binFile (@binFiles)
  {
    move($binFile, $templateDir) || die("failed to move file", $!);
  }

  # for big-endian platforms, instantiate appropriate msb files
  if ($Config{byteorder} eq '87654321' ||
      $Config{byteorder} eq '4321')
  {
    instantiateXAGMsbFilesForPorts();
  }
  else
  {
    trace("Platform is little-endian");
  }

  # move xag_inventory to $XAG_HOME/install dir and change permission
  my $invFile = catfile($XAG_HOME, 'xag_inventory');
  move(catfile($binDir, 'xag_inventory'), $invFile);
  s_set_perms(444, $invFile);
  s_set_ownergroup($CFG->params('ORACLE_OWNER'), 
                   $CFG->params('ORA_DBA_GROUP'), $invFile);

  return SUCCESS;
}

################################################################################
# Function: Instantiate XAG msb files on sparc/IBM
################################################################################
sub instantiateXAGMsbFilesForPorts
{
  trace("Entered instantiateMsbFilesForPorts");
  my $XAG_HOME = catdir($CFG->ORA_CRS_HOME, 'xag');
  my $msbDir = catdir($XAG_HOME, 'mesg');
  my @msbFiles = glob(catfile($msbDir, "*.msb"));
  
  #rename current .msb files to .le
  foreach my $msbFile (@msbFiles)
  {
    my ($fname, $fpath, $fext) =  fileparse($msbFile,'\..*');
    my $newFilePath = catfile(dirname($msbFile), "$fname.le");
    
    trace("renaming $msbFile to $newFilePath");
    if (!move($msbFile, $newFilePath))
    {
      my $errMsg = $!;
      trace("Error in moving $msbFile to $newFilePath:$errMsg");
      return FAILED;
    }
  } 
  
  #rename 'be' files to .msb so that they are in use
  #verify that the file is not stale by comparing the 
  #sizes of the default and big-endian versions
  my @beMsbFiles = glob(catfile($msbDir, "*.be"));
  foreach my $beMsbFile (@beMsbFiles)
  {
    my ($fname, $fpath, $fext) =  fileparse($beMsbFile,'\..*');
    my $newFilePath = catfile(dirname($beMsbFile), "$fname.msb");
    my $leMsbFile = catfile(dirname($beMsbFile), "$fname.le");

    my $beMsbFileSize = -s $beMsbFile;
    my $leMsbFileSize = -s $leMsbFile;
    trace("$beMsbFile size = $beMsbFileSize");
    trace("$leMsbFile size = $leMsbFileSize");

    if ($beMsbFileSize != $leMsbFileSize)
    {
      trace("Warning: $fname.msb for this platform is not latest, contact Oracle Support");
    }

    trace("renaming $beMsbFile to $newFilePath");
    if (!move($beMsbFile, $newFilePath))
    {
      trace("Error in moving $beMsbFile to $newFilePath");
      return FAILED;
    }
  }
  
  trace("instantiateMsbFilesForPorts() done");
  
  return SUCCESS;
}

1;
__END__

=head2 <sub-name>


=head3 Parameters


=head3 Returns


=head3 Usage


=cut


